/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "main.h"

typedef struct appdata {
	Evas_Object *win;
} appdata_s;

typedef struct _Item_Data
{
	int index;
	Elm_Object_Item *item;
} Item_Data;

const char *list_names[] = {"Clock", "Sound", "Display", "Bluetooth", "Reset"};

const char *icon_names[] = {"clock", "sound", "display", "bluetooth", "reset"};

static Evas_Object *
create_label(Evas_Object *parent, char *text);

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static Eina_Bool
nf_it_pop_cb(void *data, Elm_Object_Item *it)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
	return EINA_FALSE;
}

static void
nf_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	elm_naviframe_item_pop(obj);
}

static Evas_Object*
create_icon(Evas_Object *parent, int count)
{
	char buf[255];
	Evas_Object *img = elm_image_add(parent);

	snprintf(buf, sizeof(buf), "%s/settings_%s.png", ICON_DIR, icon_names[count]);
	elm_image_file_set(img, buf, NULL);
	evas_object_size_hint_min_set(img, 56, 56);
	evas_object_size_hint_max_set(img, 56, 56);
	return img;
}

static void
gl_del(void *data, Evas_Object *obj EINA_UNUSED)
{
	/* FIXME: Unrealized callback can be called after this. */
	/* Accessing Item_Data can be dangerous on unrealized callback. */
	Item_Data *id = data;
	free(id);
}

static char *
gl_text_get(void *data, Evas_Object *obj, const char *part)
{
	char buf[1024];
	Item_Data *id = (Item_Data *)data;
	int index = id->index;

	if (part && !strcmp(part, "elm.text")) {
		snprintf(buf, 1023, "%s", list_names[index]);
		return strdup(buf);
	}

	return NULL;
}

static Evas_Object *
gl_icon_get(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *icon = NULL;
	Item_Data *id = (Item_Data *)data;

	if (part && !strcmp(part, "elm.icon")) {
		icon = create_icon(obj, id->index);
		return icon;
	}

	return NULL;
}

static Evas_Object *
gl_indicator_get(void *data, Evas_Object *obj, const char *part)
{
	Evas_Object *layout;
	Evas_Object *bt_icon, *battery_icon;

	layout = elm_layout_add(obj);
	elm_layout_file_set(layout, ELM_DEMO_EDJ, "setting/indicator");
	evas_object_size_hint_weight_set(layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);

	bt_icon = elm_image_add(layout);
	elm_image_file_set(bt_icon, ICON_DIR "/bluetooth_off.png", NULL);
	elm_object_part_content_set(layout, "img.bluetooth", bt_icon);

	elm_object_part_text_set(layout, "txt.battery", "50%");

	battery_icon = elm_image_add(layout);
	elm_image_file_set(battery_icon, ICON_DIR "/battery_50.png", NULL);
	elm_object_part_content_set(layout, "img.battery", battery_icon);

	evas_object_show(layout);

	return layout;
}

static void
gl_loaded_cb(void *data EINA_UNUSED, Evas_Object *obj, void *event_info EINA_UNUSED)
{
}

static void
gl_realized_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	/* If you need, you can get item's data. */
	// Elm_Object_Item *it = event_info;
	// Item_Data *id = elm_object_item_data_get(it);
}

static void
gl_longpressed_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	/* If you need, you can get item's data. */
	// Elm_Object_Item *it = event_info;
	// Item_Data *id = elm_object_item_data_get(it);
}

static void
gl_selected_cb(void *data, Evas_Object *obj EINA_UNUSED, void *event_info)
{
	Evas_Object *nf = (Evas_Object *)data, *label;
	Elm_Object_Item *it = event_info;
	Item_Data *id = elm_object_item_data_get(it);

	/* Unhighlight item */
	elm_genlist_item_selected_set(id->item, EINA_FALSE);

	/* Create second view */
	label = create_label(nf, "This is sample view");
	elm_naviframe_item_push(nf, "Second View", NULL, NULL, label, NULL);
}

static void
gl_contracted_cb(void *data EINA_UNUSED, Evas_Object *obj EINA_UNUSED, void *event_info)
{
   Elm_Object_Item *it = event_info;
   elm_genlist_item_subitems_clear(it);
}

static Evas_Object *
create_genlist(Evas_Object *parent)
{
	Evas_Object *genlist;
	Elm_Object_Item *item;
	int n_items = NUM_OF_ITEMS;
	int index = 0;

	/* Create item class */
	elm_theme_extension_add(NULL, ELM_DEMO_EDJ);
	Elm_Genlist_Item_Class *itc_tmp = elm_genlist_item_class_new();
	itc_tmp->item_style = "setting_indicator";
	itc_tmp->func.content_get = gl_indicator_get;
	itc_tmp->func.del = gl_del;

	Elm_Genlist_Item_Class *itc = elm_genlist_item_class_new();
	itc->item_style = "1text.1icon";
	itc->func.text_get = gl_text_get;
	itc->func.content_get = gl_icon_get;
	itc->func.del = gl_del;

	genlist = elm_genlist_add(parent);

	elm_scroller_single_direction_set(genlist, ELM_SCROLLER_SINGLE_DIRECTION_HARD);

	/* HOMOGENEOUS MODE
		If item height is same when each style name is same, Use homogeneous mode. */
	elm_genlist_homogeneous_set(genlist, EINA_TRUE);

	/* COMPRESS MODE
		For the Mobile view, because it has full window, compresse mode should be used. */
	elm_genlist_mode_set(genlist, ELM_LIST_COMPRESS);

	/* Add Smart Callback */
	evas_object_smart_callback_add(genlist, "realized", gl_realized_cb, NULL);
	evas_object_smart_callback_add(genlist, "loaded", gl_loaded_cb, NULL);
	evas_object_smart_callback_add(genlist, "longpressed", gl_longpressed_cb, NULL);
	evas_object_smart_callback_add(genlist, "contracted", gl_contracted_cb, NULL);

	Item_Data *id_indi = calloc(sizeof(Item_Data), 1);
	id_indi->index = index;
	id_indi->item = elm_genlist_item_append(genlist,
												itc_tmp,
												id_indi,
												NULL,
												ELM_GENLIST_ITEM_NONE,
												NULL,
												NULL);
	elm_genlist_item_select_mode_set(id_indi->item, ELM_OBJECT_SELECT_MODE_DISPLAY_ONLY);

	for (index = 0; index < n_items; index++) {
		Item_Data *id = calloc(sizeof(Item_Data), 1);
		id->index = index;
		item = elm_genlist_item_append(genlist,
										itc,
										id,
										NULL,
										ELM_GENLIST_ITEM_NONE,
										gl_selected_cb,
										parent);
		id->item = item;
	}
	elm_genlist_item_class_free(itc_tmp);
	elm_genlist_item_class_free(itc);
	evas_object_show(genlist);

	return genlist;
}

static Evas_Object *
create_label(Evas_Object *parent, char *text)
{
	Evas_Object *label;
	label = elm_label_add(parent);

	if (text) {
		elm_object_text_set(label, text);
	}
	evas_object_show(label);

	return label;
}

static void
create_base_gui(appdata_s *ad)
{
	Elm_Object_Item *nf_it;
	Evas_Object *conform, *nf, *genlist;

	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	conform =  elm_conformant_add(ad->win);
	elm_win_resize_object_add(ad->win, conform);
	evas_object_show(conform);

	/* Naviframe */
	nf = elm_naviframe_add(conform);
	eext_object_event_callback_add(nf, EEXT_CALLBACK_BACK, nf_back_cb, ad);
	elm_object_content_set(conform, nf);
	evas_object_show(nf);

	/* Genlist */
	genlist = create_genlist(nf);

	nf_it = elm_naviframe_item_push(nf, "Settings", NULL, NULL, genlist, "empty");
	elm_naviframe_item_pop_cb_set(nf_it, nf_it_pop_cb, ad);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
		Initialize UI resources and application's data
		If this function returns true, the main loop of application starts
		If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
