/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "model/notification-control.h"
#include <badge.h>
#include "model/constants.h"
#include "utils/config.h"
#include "utils/logger.h"
#include <stdio.h>
#include <notification.h>
#include <Eina.h>

enum notification_item
{
    NOTIFICATION,
    NOTIFICATION_NORMAL,
    NOTIFICATION_BY_APP_CONTROL,
    NOTIFICATION_BY_APP_ID = 4,
};
static notification_data *notification_data_get(enum notification_item type);
static bool delete_notification_items();

static notification_h create_notification(const char *icon_path, const char *title_text, const char *content_text,
        const char *sound_path)
{
    notification_h notify = notification_create(NOTIFICATION_TYPE_NOTI);

    if (notify)
    {
        notification_set_image(notify, NOTIFICATION_IMAGE_TYPE_ICON, icon_path);
        notification_set_text(notify, NOTIFICATION_TEXT_TYPE_TITLE, title_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
        notification_set_text(notify, NOTIFICATION_TEXT_TYPE_CONTENT, content_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
        notification_set_sound(notify, NOTIFICATION_SOUND_TYPE_USER_DATA, sound_path);
    }
    return notify;
}

static void notify_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");

    delete_notification_items();

    notify_data->notification = create_notification(MAIN_MENU_PATH, title_text, alert_text, NULL);
    if (notify_data->notification)
    {
        launch_arguments_set(notify_data->notification, app_message_text, PACKAGE, NULL, NULL, NULL, NULL, NULL);
        notification_post(notify_data->notification);
    }

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br>",
            result_message, (notify_data->notification) ?
            result_message_success_text : result_message_failure_text, notify_message_text,
            alert_text,launch_argument_text, app_message_text);
}

static void notify_normal_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");

    unsigned int count = 0;
    delete_notification_items();

    notify_data->notification = create_notification(ICON2_PATH, title_text, alert_text, SOUND_PATH);
    if (notify_data->notification)
    {
        launch_arguments_set(notify_data->notification, app_message_text, PACKAGE, NULL, NULL, NULL, NULL, NULL);
        notification_set_image(notify_data->notification, NOTIFICATION_IMAGE_TYPE_BACKGROUND, THUMBNAIL_PATH);
        notification_set_text(notify_data->notification, NOTIFICATION_TEXT_TYPE_INFO_2, second_length, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
        notification_set_text(notify_data->notification, NOTIFICATION_TEXT_TYPE_INFO_SUB_2, line_text, NULL, NOTIFICATION_VARIABLE_TYPE_NONE);
        notification_post(notify_data->notification);
        increase_badge(&count);
    }

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br><br>%s<br>%d<br>",
            result_message, (notify_data->notification) ?
            result_message_success_text : result_message_failure_text, notify_message_text,
            alert_text,launch_argument_text, app_message_text, badge_is_text, count);
}
static void notify_by_app_control_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");

    delete_notification_items();

    notify_data->notification = create_notification(ICON2_PATH, notification_app, app_control, NULL);
    if (notify_data->notification)
    {
        const char *array_result[] = { result_1_text, result_2_text, NULL };
        const char *array_key[] = { SERVICE_DATA_TEXT, SERVICE_DATA_TO, NULL };
        launch_arguments_set(notify_data->notification, app_message_text,
                PACKAGE, uri_text, mime_text, operation_id_text, array_result,
                array_key);
        notification_post(notify_data->notification);
    }

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br><br>%s<br>%s<br>", result_message, (notify_data->notification) ?
            result_message_success_text : result_message_failure_text, notify_message_text,
            app_control,operation_id_is_text, operation_id_text, uri_is_text,uri_text);
}

static bool delete_notification (enum notification_item type)
{
    int err = NOTIFICATION_ERROR_NONE;
    notification_data *data = notification_data_get(type);
    if (data && data->notification)
    {
        err = notification_delete(data->notification);
        data->notification = NULL;
    }

    return (err == NOTIFICATION_ERROR_NONE);
}

static bool delete_notification_items()
{
    bool is_success = true;
    is_success = delete_notification(NOTIFICATION);
    is_success = delete_notification(NOTIFICATION_NORMAL) && is_success;
    is_success = delete_notification(NOTIFICATION_BY_APP_CONTROL) && is_success;

    return is_success;
}

static bool delete_notification_by_app_id()
{
    return delete_notification(NOTIFICATION_BY_APP_ID);
}

void delete_all_notifications()
{
    delete_notification(NOTIFICATION);
    delete_notification(NOTIFICATION_NORMAL);
    delete_notification(NOTIFICATION_BY_APP_CONTROL);
    delete_notification(NOTIFICATION_BY_APP_ID);
}

static void remove_notification_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");
    bool is_success = delete_notification_items();

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>", result_message, (is_success) ?
            result_message_success_text : result_message_failure_text, remove_text);

    badge_remove(PACKAGE);
}
static void notify_by_app_id_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");

    delete_notification_by_app_id();

    notify_data->notification = create_notification(ICON2_PATH, title_text, notify_with_request, NULL);
    if (notify_data->notification)
    {
        launch_arguments_set(notify_data->notification, app_message_text, SETTING_PKGNAME, NULL, NULL, NULL, NULL, NULL);
        notification_post(notify_data->notification);
    }

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%s<br><br>%s<br>%s<br>", result_message,
            (notify_data->notification) ? result_message_success_text : result_message_failure_text,
            app_id_is_text, SETTING_PKGNAME, icon_path_is_text, ICON2_PATH);
}
static void remove_notification_by_app_id_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");
    bool is_success = delete_notification_by_app_id();

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br><br>%s<br>%s<br>", result_message, (is_success) ?
            result_message_success_text : result_message_failure_text, remove_text, target_app_id_text, SETTING_PKGNAME);
}
static void set_badge_number_by_app_id_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");

    unsigned int count = 0;
    bool is_success = increase_badge(&count);

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br><br>%s<br>%d<br><br>%s<br>%s<br>", result_message, (is_success) ?
            result_message_success_text : result_message_failure_text, badge_is_text, count, app_id_is_text, PACKAGE);
}
static void remove_badge_number_by_app_id_cb(notification_data *notify_data)
{
    RETM_IF(!notify_data, "notify_data is NULL");

    unsigned int count = 0;
    bool is_success = decrease_badge(&count);

    snprintf(notify_data->result_text, TEXT_MAX_LEN, "%s%s<br> <br>%s<br>%d<br> <br>%s<br>%s<br>", result_message, (is_success) ?
            result_message_success_text : result_message_failure_text, badge_is_text, count, app_id_is_text, PACKAGE);
}

static notification_data *notification_data_get(enum notification_item type)
{
    int size = 0;
    notification_data *data = NULL;
    notification_data *list = (notification_data *) notification_list_get(
            &size);
    if (type < size)
    {
        data = &list[type];
    }
    return data;
}

notification_data *notification_list_get(int *size)
{
    static notification_data notification_list[] =
    {
            {"Notify", NULL, 0, 0, notify_cb, },
            {"Notify(normal)", NULL,  0, 0, notify_normal_cb, },
            {"NotifyByAppControl", NULL,  0, 0, notify_by_app_control_cb, },
            {"RemoveNotification", NULL,  0, 0, remove_notification_cb, },
            {"NotifyByAppId", NULL,  0, 0, notify_by_app_id_cb, },
            {"RemoveNotificationByAppId", NULL,  0, 0, remove_notification_by_app_id_cb, },
            {"SetBadgeNumberByAppId", NULL,  0, 0, set_badge_number_by_app_id_cb, },
            {"RemoveBadgeNumberByAppId", NULL,  0, 0, remove_badge_number_by_app_id_cb, }
    };
    *size = sizeof(notification_list) / sizeof(notification_list[0]);
    return notification_list;
}

