/*
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*jslint devel: true */
/*global window, document, setTimeout, tizen, $, app, TemplateManager,
 Helpers */

/**
 * @class Ui
 */
function Ui() {
    'use strict';
}

(function start() { // strict mode wrapper
    'use strict';
    Ui.prototype = {
        /**
         * root mode
         * @type {boolean}
         */
        root: true,

        /**
         * locked folders
         * @type {array}
         */
        lockedFolders: ['ringtones'],

        /**
         * UI edit mode
         * @type {boolean}
         */
        editMode: false,

        /**
         * @type {TemplateManager}
         */
        templateManager: null,

        /**
         * @type Helpers
         */
        helpers: null,

        /**
         * name of row gradient class
         */
        CSS_GRADIENT_CLASS: 'gradientBackground',

        /**
         * Edit tabbar actions
         * @type {number}
         */
        EDIT_TABBAR_DELETE_ACTION: 0,
        EDIT_TABBAR_MOVE_ACTION: 1,
        EDIT_TABBAR_COPY_ACTION: 2,
        EDIT_TABBAR_CANCEL_ACTION: 3,

        currentHeaderHeight: null,
        currentScrollPosition: null,

        /**
         * UI Initialization
         */
        init: function Ui_init(storages) {
            this.templateManager = new TemplateManager();
            this.helpers = new Helpers();
            // Disable text selection
            $.mobile.tizen.disableSelection(document);
            $(document).ready(this.initDom.bind(this, storages));
        },

        initDom: function Ui_initDom(storages) {
            var self = this;

            this.templateManager.loadToCache(
                [
                    'fileRow',
                    'folderRow',
                    'levelUpRow',
                    'emptyFolder'
                ],
                function success() {
                    self.addEvents();
                    self.displayStorages(storages);
                }
            );

            $('#infoPopup').popup();
        },

        /**
         * Add UI events
         */
        addEvents: function Ui_addEvents() {
            var self = this;
            document.addEventListener('webkitvisibilitychange',
                function onWebkitVisibilityChange() {
                    if (document.webkitVisibilityState === 'visible') {
                        if (self.isOperationInProgress()) {
                            return;
                        }
                        self.refreshSelectAllStatus();
                        app.refreshCurrentPage(true);
                    }
                });

            window.addEventListener('tizenhwkey', function onTizenhwkey(e) {
                var uri = $('#navbar span+span').attr('uri');
                if (e.keyName === 'back') {
                    if (self.isOperationInProgress()) {
                        return;
                    } else if ($.mobile.popup.active) {
                        $.mobile.popup.active.close();
                    } else if (self.editMode === true) {
                        self.handleCancelEditAction();
                    } else if (!uri) {
                        if (app.ui.root === false) {
                            app.ui.root = true;
                            $('#fileList').empty();
                            app.ui.appendFolderRow(0, 'root');
                            app.currentPath = 'root';
                        } else {
                            tizen.application.getCurrentApplication().exit();
                        }
                    } else if (!app.ui.root) {
                        app.goLevelUp();
                    }
                }
            });

            $(window).resize(function onResize() {
                $.mobile.activePage.page('refresh');
            });

            // touch events for all nodes
            $('ul#fileList')
                .on('tap', 'li.levelUp', function onClick() {
                    if (self.editMode === true) {
                        self.handleCancelEditAction();
                    }
                    app.goLevelUp();
                })
                .on('tap', 'li.node', function onClick(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.handleNodeClick($(this), true);
                })
                .on('change', 'input[type=checkbox]', function onChange(e) {
                    self.handleNodeClick($(this).closest('li.node'), false);
                })
                .on('touchstart', 'li', function onTouchstart(event) {
                    $(this).addClass(self.CSS_GRADIENT_CLASS);
                })
                .on('touchend touchmove', 'li', function onTouchend(event) {
                    $(this).removeClass(self.CSS_GRADIENT_CLASS);
                });

            $('.selectAll input').on(
                'change',
                this.handleSelectAllChange.bind(this)
            );

            // navbar
            $('#navbar').on('tap', 'span', function onClick() {
                var uri = $(this).attr('uri');
                if (uri === 'home' && app.currentPath !== '') {
                    app.displayStorages();
                } else if (uri === app.model.currentPath) {
                    app.displayFolder(uri, true);
                } else {
                    if (self.editMode === true) {
                        self.handleCancelEditAction();
                    }
                    app.displayFolder(uri);
                }
            });

            // level up
            $('#levelUpBtn').on('tap', function onClick() {
                if (self.editMode === true) {
                    self.handleCancelEditAction();
                }
                app.goLevelUp();
            });

            $('#homeBtn').on('tap', app.displayStorages.bind(app));

            // edit action
            $('#editActionBtn').on('tap', this.handleEditAction.bind(this));

            // delete action
            $('#deleteActionBtn').on(
                'tap',
                this.handleDeleteAction.bind(this)
            );

            // copy action
            $('#copyActionBtn').on('tap', this.handleCopyAction.bind(this));

            // move action
            $('#moveActionBtn').on('tap', this.handleMoveAction.bind(this));

            // paste action
            $('a#pasteActionBtn').on('tap', function onClick() {
                $('#morePopup').popup('close');
                $('#infoPopup')
                    .popup('open')
                    .one('popupafteropen', function afterOpen() {
                        app.pasteClipboard();
                    });
            });

            // remove active class
            $('[data-role = "tabbar"] li > a').on('tap', function onClick() {
                $(this).removeClass('ui-focus, ui-btn-active');
            });
            // exit
            $('.ui-myExit').on('tap', app.exit);

            // add folder popup actions
            $('#addFolderPopup').on('popupafterclose',
                function onPopupAfterClose() {
                    // clear input value
                    $('#newFolderName').val('New folder');
                });

            $('#newFolderName').on('focus', function onFocus() {
                if ($(this).attr('value') === 'New folder') {
                    $(this).attr('value', '');
                }
            });

            $('#saveNewFolder').on('tap', this.saveNewFolder.bind(this));
            $('#newFolderForm').on('submit', this.saveNewFolder.bind(this));
        },

        saveNewFolder: function Ui_saveNewFolder(e) {
            var folderName = $('#newFolderName').val(), status = true,
                open = function open() {
                    $('#addFolderPopup').popup('open', {
                        positionTo: 'window'
                    });
                },
                self = this,
                aPopup;
            e.preventDefault();
            e.stopPropagation();

            $('#addFolderPopup').one('popupafterclose',
                function onPopupAfterClose() {
                    if (folderName === '') {
                        self.alertPopup('Empty folder name', open);
                        status = false;
                    } else if (folderName.match(/[\*\.\/\\\?\"\'\:<>|]/)) {
                        self.alertPopup('The following special characters '  +
                            'are not allowed: *./\\?:<>|\'"', open);
                        status = false;
                    } else {
                        status = app.createDir(folderName, open);
                    }
                });
            aPopup = $.mobile.popup.active;
            if (
                aPopup && aPopup.element.getAttribute('id') === 'addFolderPopup'
            ) {
                aPopup.close();
            }
            return status;
        },

        /**
         * Returns if some operation (i.e. copying/moving nodes) is in progress
         * This is detected by infoPopup visibility
         * @return {boolean}
         */
        isOperationInProgress: function Ui_isOperationInProgress() {
            var activePopup = $.mobile.popup.active;

            if (activePopup && activePopup.id === 'infoPopup') {
                return true;
            } else {
                return false;
            }
        },

        /**
         * Closes popup with information about operations in progress
         */
        closeInfoPopup: function Ui_closeInfoPopup() {
            $('#infoPopup').popup('close');
        },

        /**
         * Closes active popup
         */
        closeActivePopup: function Ui_closeActivePopup() {
            var popup = $.mobile.popup.active;
            if (popup) {
                popup.close();
            }
        },

        alertPopup: function Ui_alertPopup(text, callback) {
            $('#alertPopup .text').text(text);
            $('#alertPopup').popup('open', {'positionTo': 'window'});
            if (callback instanceof Function) {
                $('#alertPopup').one('popupafterclose',
                    function onPopupafterclose() {
                        callback();
                    });
            }
        },

        confirmPopup: function Ui_confirmPopup(text, confirmCallback,
                                            completeCallback) {
            var popup = $('#confirmPopup');
            popup.find('.text').text(text);
            popup.popup('open', {positionTo: 'window'});
            popup.find('.confirm').one('vclick', function onClick() {
                if (confirmCallback instanceof Function) {
                    confirmCallback();
                }
            });
            if (completeCallback instanceof Function) {
                popup.one('popupafterclose', function onPopupAfterClose() {
                    completeCallback();
                });
            }
        },

        /**
         * Handler for node click
         * @param {File} node
         * @param {boolean} toggleCheckbox
         */
        handleNodeClick: function Ui_handleNodeClick(node, toggleCheckbox) {
            if (this.root) {
                app.model.loadInternalStorages(function success() {
                    app.displayStorages();
                });
                this.root = false;
            } else if (this.editMode === true) {
                if (toggleCheckbox === true) {
                    this.toggleCheckBoxState(node); // select the checkbox
                }

                this.refreshSelectAllStatus();
                this.refreshEditMenu();
            } else if (node.hasClass('folder')) {
                // otherwise display folder
                app.displayFolder(node.attr('uri'));
            } else {
                // file
                app.openFile(node.attr('fullUri'));
            }
        },

        /**
         * Handler for edit action
         */
        handleEditAction: function Ui_handleEditAction() {
            this.editMode = true;
            $('.standardTabbar').hide();
            $('div.editTabbar').show();
            this.disableControlBarButtons(
                $('div.editTabbar'),
                [
                    this.EDIT_TABBAR_DELETE_ACTION,
                    this.EDIT_TABBAR_COPY_ACTION,
                    this.EDIT_TABBAR_MOVE_ACTION
                ]
            );
            $('#fileList .folder .nodename, #fileList > li > span.nodename')
                .animate({'width': '70%'});
            this.showEditCheckBoxes();
        },

        /**
         * Handler for cancel edit action
         */
        handleCancelEditAction: function Ui_handleCancelEditAction(emptyList) {
            this.editMode = false;
            $('div.editTabbar').hide();
            $('.standardTabbar').show();
            $('#fileList .folder .nodename,  #fileList > li > span.nodename')
                .animate({'width': '75%'});
            this.hideEditCheckBoxes();
            if (this.isFileListEmpty() || emptyList) {
                $('#editActionBtn').addClass('vhidden').blur();
            }
        },

        /**
         * Handler for delete action
         */
        handleDeleteAction: function Ui_handleDeleteAction(e) {
            var nodesToDelete = [],
                $rowElement,
                self = this,
                listLength = $('ul#fileList li.node').length;

            e.stopPropagation();
            e.preventDefault();

            this.confirmPopup('Selected nodes will be deleted. Are you sure?',
                function success() {
                    $('ul#fileList input:checkbox:checked')
                        .each(function pushElements(index) {
                            $rowElement = $(this).closest('li');
                            nodesToDelete.push({
                                id: $rowElement.attr('id'),
                                uri: $rowElement.attr('uri'),
                                name: $rowElement.attr('label'),
                                folder: $rowElement.hasClass('folder')
                            });
                        });

                    if (nodesToDelete.length > 0) {
                        app.deleteNodes(nodesToDelete);
                        self.scrollContentTo(0);
                        $('ul#fileList input:checkbox:checked').remove();
                        if (nodesToDelete.length === listLength) {
                            self.handleCancelEditAction(true);
                        } else {
                            self.refreshEditMenu();
                        }
                    }
                },
                function error() {
                }
            );
        },

        /**
         * Handler for copy action
         */
        handleCopyAction: function Ui_handleCopyAction(e) {
            var paths = [];

            e.stopPropagation();
            e.preventDefault();

            if (this.editMode === true) {
                $('ul#fileList input:checkbox:checked').each(
                    function pushElements() {
                        paths.push($(this).closest('li').attr('uri'));
                });
                app.saveToClipboard(paths, app.clipboard.COPY_MODE_ID);
            }
        },

        /**
         * Handler for move action
         */
        handleMoveAction: function Ui_handleMoveAction(e) {
            var paths = [];

            e.stopPropagation();
            e.preventDefault();

            if (this.editMode === true) {
                $('ul#fileList input:checkbox:checked').each(
                    function pushElements(index) {
                        paths.push($(this).closest('li').attr('uri'));
                    }
                );
                app.saveToClipboard(paths, app.clipboard.MOVE_MODE_ID);
            }
        },

        /**
         * Scrolls content to the specified position
         */
        scrollContentTo: function scrollContentTo(value) {
            $('#main [data-role="content"]').scrollview('scrollTo', 0, value);
        },

        /**
         * @param {FileSystemStorage[]} nodes Storage elements
         */
        displayStorages: function Ui_displayStorages(nodes) {
            var len = nodes.length, nodeName, i;

            this.updateNavbar('');
            $('#fileList').empty();

            for (i = 0; i < len; i = i + 1) {
                nodeName = nodes[i].label;
                if (nodeName !== '' &&
                    (nodes[i].type === 0 || nodes[i].type === 'INTERNAL') &&
                    nodeName.indexOf('wgt-') === -1 &&
                    $.inArray(nodeName, this.lockedFolders) === -1) {
                    this.appendFolderRow(i, nodeName);
                }
            }

            $('#levelUpBtn, #homeBtn').addClass('vhidden');
            $('#editActionBtn, #moreActionBtn').addClass('vhidden').blur();

            $('h1#mainTitle').html('Media');

            if (this.editMode) {
                this.scrollContentTo(0);
            }

            this.hideSelectAllArea();
            this.handleCancelEditAction();
        },

        appendFolderRow: function appendFolderRow(id, name) {
            $(this.templateManager.get('folderRow', {
                id: id,
                name: name,
                uri: name,
                fullUri: name,
            })).appendTo('#fileList');
        },

        /**
         * renders node list for folder
         * @param {string} folderName
         * @param {File[]} nodes
         * @param {bool} [refresh=false]
         */
        /*jshint maxcomplexity:15, maxstatements:49 */
        displayFolder: function Ui_displayFolder(folderName, nodes, refresh) {
            var len = nodes.length,
                listElements = [this.templateManager.get('levelUpRow')],
                nodeName,
                checkedRows = [],
                checkedRowsLen,
                i;
            refresh = refresh || false;

            // update title
            this.updateTitle(
                this.templateManager.modifiers.escape(folderName)
            );
            // update navbar
            this.updateNavbar(
                this.templateManager.modifiers.escape(folderName)
            );
            this.refreshPasteActionBtn();

            nodes.sort(this.helpers.fileComparison);

            // render nodes
            for (i = 0; i < len; i = i + 1) {
                nodeName = nodes[i].name;
                if (nodeName !== '') {
                    if (nodes[i].isDirectory) {
                        // folder
                        listElements.push(this.templateManager.get(
                            'folderRow',
                            {
                                id: i,
                                name: nodeName,
                                uri: nodes[i].fullPath,
                                fullUri: nodes[i].toURI()
                            }
                        ));
                    } else {
                        // file
                        listElements.push(this.templateManager.get(
                            'fileRow',
                            {
                                id: i,
                                name: nodeName,
                                uri: nodes[i].fullPath,
                                fullUri: nodes[i].toURI(),
                                thumbnailURI: this.helpers.getThumbnailURI(
                                    nodeName,
                                    nodes[i]
                                )
                            }
                        ));
                    }
                }
            }

            if (listElements.length === 1) {
                // set content for empty folder
                listElements.push(this.templateManager.get('emptyFolder'));
                // hide edit button for empty content
                $('#editActionBtn').addClass('vhidden').blur();
                this.handleCancelEditAction();
            } else {
                $('#editActionBtn').removeClass('vhidden');
            }

            // scroll to top of list
            this.scrollContentTo(0);

            $('#levelUpBtn').removeClass('vhidden');
            $('#homeBtn').removeClass('vhidden');
            $('#moreActionBtn').removeClass('vhidden');

            if (refresh === true && this.editMode === true) {
                $.each($('#fileList .ui-checkbox input:checked'),
                    function pushElements() {
                        checkedRows.push($(this).closest('li').attr('uri'));
                    });
            }

            // update file list
            $('#fileList').html(listElements.join(''))
                .trigger('refresh')
                .trigger('create');

            if (this.editMode === true) {
                $('.selectAll').show();
                $('#fileList .folder .nodename, #fileList > li > span.nodename')
                    .css('width', '70%');
                $('ul#fileList > li').css('paddingLeft', '0.5rem');
                $('.my-ui-checkbox').removeClass('hidden');
                if (refresh === true) {
                    checkedRowsLen = checkedRows.length;
                    if (checkedRowsLen) {
                        if (
                            checkedRowsLen !==
                                $('#fileList .ui-checkbox input').length
                        ) {
                            this.setCheckboxValue('.selectAll input', false);
                        }
                        // restore checked checkboxes
                        for (i = 0; i < checkedRowsLen; i += 1) {
                            this.setCheckboxValue(
                                '#' +
                                    $('#fileList [uri="' +
                                    checkedRows[i] +
                                    '"]').attr('id') +
                                    ' input:checkbox',
                                'checked'
                            );
                        }

                        // if there are no checked checkboxes
                        if (!$('#fileList .ui-checkbox input:checked').length) {
                            this.clearDeleteMode();
                        }
                    } else {
                        this.clearDeleteMode();
                    }
                }
            } else {
                $('.selectAll').hide();
                $('#fileList .folder .nodename, #fileList > li > span.nodename')
                    .css('width', '75%');
                $('ul#fileList > li').css('paddingLeft', '0');
                $('.my-ui-checkbox').addClass('hidden');
                this.clearDeleteMode();
            }

            if (!refresh) {
                this.hideSelectAllArea();
            }

            this.refreshSelectAllStatus();
        },

        /**
         * Clear confirm popup and disable action buttons
         */
        clearDeleteMode: function Ui_clearDeleteMode() {
            var aPopup = $.mobile.popup.active;
            if (aPopup && aPopup.element.getAttribute('id') ===
                'confirmPopup') {
                $.mobile.popup.active.close();
            }
            this.disableControlBarButtons(
                $('div.editTabbar'),
                [
                    this.EDIT_TABBAR_DELETE_ACTION,
                    this.EDIT_TABBAR_COPY_ACTION,
                    this.EDIT_TABBAR_MOVE_ACTION
                ]
            );
        },

        /**
         * Toggle a checkbox associated with a given list element
         * @param {jQuery} listElement
         */
        toggleCheckBoxState: function Ui_toggleCheckBoxState(listElement) {
            var checkboxInput = null;

            checkboxInput = listElement.find('form > div.ui-checkbox input');
            this.setCheckboxValue(
                checkboxInput,
                !checkboxInput.prop('checked')
            );
        },

        /**
         * Shows item checkboxes and topbar with select all option
         */
        showEditCheckBoxes: function Ui_showEditCheckBoxes() {
            var self = this;

            this.showSelectAllArea();

            $('ul#fileList > li').animate(
                {paddingLeft: '0.5rem'},
                500,
                'swing',
                function removeClassHidden() {
                    self.editMode = true;
                    $('.my-ui-checkbox').removeClass('hidden');
                }
            );
        },

        /**
         * Hides item checkboxes and topbar with select all option
         * All checkboxes are auto uncheked
         */
        hideEditCheckBoxes: function Ui_hideEditCheckBoxes() {
            var self = this;

            this.hideSelectAllArea(); // hide select all option topbar

            $('ul#fileList > li').animate(
                {paddingLeft: '0'},
                200,
                'swing',
                function addClassHidden() {
                    $('.my-ui-checkbox').addClass('hidden');
                    $.mobile.activePage.page('refresh');
                }
            );

            // uncheck all checkboxes
            $('ul#fileList input[type=checkbox]').each(
                function setCheckboxValue(index) {
                    self.setCheckboxValue(this, false);
                }
            );

            this.setCheckboxValue('.ui-header .selectAll input', false);
        },

        /**
         * Save current header and content height
         */
        saveHeights: function Ui_saveHeights() {
            this.currentHeaderHeight = $('#main div[data-role="header"]')
                .height();
            this.currentScrollPosition = $('#main div[data-role="content"]')
                .scrollview('getScrollPosition').y;
        },

        /**
         * Changes content scroll position after showing/hiding selectAllArea
         */
        changeContentScrollPosition: function Ui_changeContentScrollPosition() {
            var diff,
                contentHeight;
            if (this.currentScrollPosition !== 0) {
                contentHeight = $('#main div[data-role="header"]').height();
                diff = contentHeight - this.currentHeaderHeight;
                $('#main div[data-role="content"]')
                    .scrollview(
                        'scrollTo',
                        0,
                        -(this.currentScrollPosition + diff)
                    );
            }
        },

        /**
         * Shows topbar with select all option
         */
        showSelectAllArea: function Ui_showSelectAllArea() {
            this.saveHeights();
            $('.selectAll').show();
            $.mobile.activePage.page('refresh');
            this.changeContentScrollPosition();
        },

        /**
         * Hides topbar with select all option
         */
        hideSelectAllArea: function Ui_hideSelectAllArea() {
            this.saveHeights();
            $('.selectAll').hide();
            $.mobile.activePage.page('refresh');
            this.changeContentScrollPosition();
        },

        /**
         * Enable specified options for tabbar
         * @param {object} tabbar
         * @param {array} enableOptions options to enable
         */
        enableControlBarButtons: function Ui_enableControlBarButtons(
            tabbar,
            enableOptions
        ) {
            var i = 0,
                len = enableOptions.length;

            for (i = 0; i < len; i += 1) {
                tabbar.tabbar('enable', enableOptions[i]);
            }
        },

        /**
         * Disable specified options for tabbar
         * @param {object} tabbar controlbar
         * @param {array} disableOptions options to enable
         */
        disableControlBarButtons: function Ui_disableControlBarButtons(
            tabbar,
            disableOptions
        ) {
            var i = 0,
                len = disableOptions.length;

            for (i = 0; i < len; i += 1) {
                tabbar.tabbar('disable', disableOptions[i]);
            }
        },

        /**
         * @param {string} path
         */
        updateTitle: function Ui_updateTitle(path) {
            var regexp = new RegExp('([^\/])+$', 'g'),
                match = path.match(regexp),
                lastDir = match[0] || '(dir)';
            $('h1#mainTitle').html(lastDir);
        },

        /**
         * @param {string} path
         */
        updateNavbar: function Ui_updateNavbar(path) {
            var html = ['<span uri="home">Media</span>'],
                splitted,
                len,
                i;

            if (typeof path === 'string' && path !== '') {
                splitted = path.split('/');
                len = splitted.length;

                for (i = 0; i < len; i = i + 1) {
                    html.push(
                        '<span uri="' +
                            splitted.slice(0, i + 1).join('/') +
                            '">' +
                            splitted[i] +
                            '</span>'
                    );
                }
            }
            $('#navbar').html(html.join(' > '));
        },

        handleSelectAllChange: function Ui_handleSelectAllChange() {
            var $selectAllInput = $('.ui-header .selectAll .ui-checkbox input'),
                self = this;
            $selectAllInput.data('checkboxradio').refresh();

            if ($selectAllInput.is(':checked')) {
                // check all checkboxes
                $('ul#fileList input[type=checkbox]').each(
                    function setCheckboxValue(index) {
                        self.setCheckboxValue(this, true);
                    }
                );

                this.enableControlBarButtons(
                    $('.editTabbar'),
                    [
                        this.EDIT_TABBAR_DELETE_ACTION,
                        this.EDIT_TABBAR_COPY_ACTION,
                        this.EDIT_TABBAR_MOVE_ACTION
                    ]
                );
            } else {
                $('ul#fileList input[type=checkbox]').each(
                    function setCheckboxValue(index) {
                        self.setCheckboxValue(this, false);
                    }
                );

                this.disableControlBarButtons(
                    $('.editTabbar'),
                    [
                        this.EDIT_TABBAR_DELETE_ACTION,
                        this.EDIT_TABBAR_COPY_ACTION,
                        this.EDIT_TABBAR_MOVE_ACTION
                    ]
                );
            }
        },

        /**
         *
         */
        refreshSelectAllStatus: function Ui_refreshSelectAllStatus() {
            var $selectAllInput = $('.ui-header .selectAll .ui-checkbox input');
            // update status of select all checkbox
            if ($('ul#fileList input:checkbox:not(:checked)').length === 0) {
                // all nodes checked
                this.setCheckboxValue($selectAllInput, true);
            } else {
                // some node is not checked
                this.setCheckboxValue($selectAllInput, false);
            }
        },

        /**
         * Refresh activity of edit menu
         */
        refreshEditMenu: function refreshEditMenu() {
            if ($('ul#fileList input:checkbox:checked').length > 0) {
                this.enableControlBarButtons(
                    $('.editTabbar'),
                    [
                        this.EDIT_TABBAR_DELETE_ACTION,
                        this.EDIT_TABBAR_COPY_ACTION,
                        this.EDIT_TABBAR_MOVE_ACTION
                    ]
                );
            } else {
                this.disableControlBarButtons(
                    $('.editTabbar'),
                    [
                        this.EDIT_TABBAR_DELETE_ACTION,
                        this.EDIT_TABBAR_COPY_ACTION,
                        this.EDIT_TABBAR_MOVE_ACTION
                    ]
                );
            }
        },

        /**
         * Remove html node element from list
         * @param {string} nodeUri node uri
         */
        removeNodeFromList: function Ui_removeNodeFromList(nodeUri) {
            $('ul#fileList > li[uri="' + nodeUri.replace(/"/g, '\"') + '"]')
                .remove();

            // hide select All checkbox if removed all elements;
            if ($('ul#fileList > li.node').length === 0) {
                this.hideSelectAllArea();
            }
        },

        /**
         * Enable/Disable
         */
        refreshPasteActionBtn: function Ui_refreshPasteActionBtn() {
            if (app.emptyClipboard()) {
                $('#pasteActionBtnRow').addClass('hidden');
            } else {
                $('#pasteActionBtnRow').removeClass('hidden');
            }
            $('#moreActionMenu').listview('refresh');
        },

        isFileListEmpty: function Ui_isFileListEmpty() {
            return ($('ul#fileList').children('.node').length < 1);
        },

        setCheckboxValue: function Ui_setCheckboxValue(element, value) {
            var $element = $(element);

            if ($element) {
                $element.prop('checked', value);
                $element.checkboxradio('refresh');
            }
        }
    };
}());
