/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Ho Namkoong <ho.namkoong@samsung.com> 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */

package org.tizen.common.sdb;

import static org.junit.Assert.assertTrue;
import static org.tizen.common.Shell.*;
import static org.tizen.common.sdb.SdbTestUtil.*;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.MessageFormat;

import org.junit.Test;
import org.tizen.common.Pattern;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.IOUtil;

/**
 * <p>
 * InstallTest
 * 
 * Test case for sdb install, uninstall
 *
 * </p>
 * 
 * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
 */
public class InstallTest {
    
    public static int INSTALL_WAIT = 1000;
    
    
    /**
     * Tests for install with wrong package type
     * Expected output: error: unknown package type
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * @throws Exception
     */
    @Test
    public void test_installNotSupportedPkgType() throws Exception {
        final String[] wrongFormats = {"asdf", "asdf.wgk", "asdf.wgtt", "asdf.wg", "asdf.t.tpkk", "asdf.t.tp", 
                "'asdfjkl'", "\"asdfjkl\""};
        String serialNumber = getSerialNumber()[0];
        
        for(String wrongFormat: wrongFormats) {
            String[] results = runAndGetWholeString(SDB_PATH + String.format(" -s %s install %s", serialNumber, wrongFormat), true, true);
            
            String output = "";
            for(String result: results) {
                output = output + result + "\n";
            }
            
            assertTrue(String.format("output of '%s' is wrong. output: '%s'.", 
                    SDB_PATH + String.format(" -s %s install %s", serialNumber, wrongFormat), output), 
                    results.length == 1);
            
            assertTrue(String.format("wrong message for '%s'. output: '%s'.", 
                    SDB_PATH + String.format(" -s %s install %s", serialNumber, wrongFormat), output), 
                    results[0].contains("error: unknown package type"));
        }
    }
    
    @Test
    public void test_installNotExistFile() throws Exception {
        final String[] notExistFiles = {"asdf.tpk", "asdf.wgt"};
        
        String serialNumber = getSerialNumber()[0];
        
        for(String notExistFile: notExistFiles) {
            String[] results = runAndGetWholeString(SDB_PATH + String.format(" -s %s install %s", serialNumber, notExistFile), true, true);
            
            String output = "";
            for(String result: results) {
                output = output + result + "\n";
            }
            
            assertTrue(String.format("output of '%s' is wrong. output: '%s'.", 
                    SDB_PATH + String.format(" -s %s install %s", serialNumber, notExistFile), output), 
                    results.length == 1);
            
            assertTrue(String.format("wrong message for '%s'. output: '%s'.", 
                    SDB_PATH + String.format(" -s %s install %s", serialNumber, notExistFile), output), 
                    results[0].contains("No such file or directory"));
        }
    }
    
    /**
     * Test {sdb install, sdb uninstall (native)}
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * 
     * command
     * 1. extract ZFPK8xqL5F-1.0.0-i386.tpk from class path and copy it to the 'copied.tpk'
     * 2. sdb uninstall ZFPK8xqL5F (app id)
     * 3. sdb install copied.tpk
     * 4. sdb shell pkgcmd -l | grep ZFPK8xqL5F
     * 5. sdb uninstall ZFPK8xqL5F (app id)
     * 6. sdb shell pkgcmd -l | grep ZFPK8xqL5F
     * expected result
     * {0} pkgid [{1}]{2}, {1} is equal with ZFPK8xqL5F after first pkgcmd
     * {0} pkgid [{1}]{2}, {1} does not exist after second pkgcmd.
     */
    @Test
    public void test_installUninstallNative() throws Exception {
        final String tpkName = "ZFPK8xqL5F-1.0.0-i386.tpk";
        final String appId = "ZFPK8xqL5F";
        final String copiedTpkName = "copied.tpk";
        
        if(SdbTestUtil.isNativePluginInstalled()) {
            test_installUninstall(tpkName, copiedTpkName, appId);
        }
    }
    
    /**
     * Test {sdb install, sdb uninstall (web)}
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * 
     * command and expected result are same with native except app id and package file 
     */
    @Test
    public void test_installUninstallWeb() throws Exception {
        final String wgtName ="sdbTest.wgt";
        final String appId = "SA6n8VGmiM";
        final String copiedWgtName = "copied.wgt";
        
        test_installUninstall(wgtName, copiedWgtName, appId);
    }
    
    private void test_installUninstall(String pkgName, String copiedName, String appId) throws Exception {
        
        class InstallUninstallResult {
            @Pattern( pattern = "{0}pkgid [{1}]{2}", index = 1) public String appId;
        }
        

        final int bufferSize = 8192;
        
        InputStream in = null;
        OutputStream out = null;
        try {
            in = InstallTest.class.getClassLoader().getResourceAsStream(pkgName);
            in = new BufferedInputStream( in, bufferSize );
            out = new FileOutputStream( copiedName, false );
            out = new BufferedOutputStream( out, bufferSize );
            IOUtil.redirect(in, out);
            
            assertTrue("tpk file is not copied", new File(copiedName).exists());
            
            String serialNumber = getSerialNumber()[0];
            runAndWait(SDB_PATH + String.format(" -s %s uninstall %s", serialNumber, appId));
            runAndWait( SDB_PATH + String.format(" -s %s install %s", serialNumber, copiedName));
            boolean success = false;
            String errMsg = "";
            for(int i=0; i < MAX_TRIAL; i++) {
                InstallUninstallResult result = 
                        run( SDB_PATH + String.format(" -s %s shell pkgcmd -l | grep %s", serialNumber, appId)).
                        expect(new InstallUninstallResult());
                if(result == null) {
                    errMsg = "output of pkgcmd for install is null";
                }
                else if(result.appId == null) {
                    errMsg = "output of pkgcmd for install does not have appId";
                }
                else if(!result.appId.equals(appId)) {
                    errMsg = MessageFormat.format("output of pkgcmd for install {0} does not equal with {1}", result.appId, appId);
                }
                else {
                    success = true;
                    break;
                }
                Thread.sleep(INSTALL_WAIT);
            }
            assertTrue(errMsg, success);
            runAndWait( SDB_PATH + String.format(" -s %s uninstall %s", serialNumber, appId));
            success = false;
            for(int i=0; i< MAX_TRIAL; i++) {
                InstallUninstallResult result = run( SDB_PATH + String.format(" -s %s shell pkgcmd -l | grep %s", serialNumber, appId)).
                        expect(new InstallUninstallResult());
                if(result == null) {
                    errMsg = "output of pkgcmd for uninstall is null";
                }
                else if(result.appId != null) {
                    MessageFormat.format("fail to uinstall tpk file {0}", pkgName);
                }
                else {
                    success = true;
                    break;
                }
                Thread.sleep(INSTALL_WAIT);
            }
            assertTrue(errMsg, success);
        }
        finally {
            File copiedFile = new File(copiedName);
            if(copiedFile.exists()) {
                FileUtil.recursiveDelete(copiedFile);
            }
            IOUtil.tryClose(in, out);
        }
        
    }
}
