/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/package org.tizen.common.util;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.Flushable;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;

import org.junit.Test;


/**
 * IOUtilTest.
 * 
 * Test case for {@link IOUtil}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see IOUtil
 * 
 */
public class IOUtilTest
{
	/**
	 * Test {@link IOUtil#tryClose(Object...)}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link IOUtil#tryClose(Object...)}
	 */
	@Test
	public
	void
	test_tryClose()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { null },
			new Object[] { new Closeable[] { new ByteArrayOutputStream(), new ByteArrayOutputStream() } },
			new Object[] { new Closeable[] { null } },
			new Object[] { new Closeable[] { null, null, null } },
			new Object[] { new Closeable[] { null, new ByteArrayOutputStream() } },
			new Object[] { new Closeable[] { new ByteArrayOutputStream(), null } },
			new Object[] { new Closeable[] { new ByteArrayOutputStream(), null } },
			new Object[] { new Closeable[] { new ByteArrayOutputStream(), null, new ByteArrayOutputStream() } },
		};
		
		int iTestCase = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTestCase;

			try
			{
				IOUtil.tryClose( TEST_CASE[0] );
			}
			catch ( final Throwable e )
			{
				fail( iTestCase + " th test case failed" );
			}
		}

		/* Mock Object using Mockito */
		InputStream in = mock( InputStream.class );
		IOUtil.tryClose( in );
		verify( in ).close();
	}

	/**
	 * Test {@link IOUtil#tryFlush(Flushable...)}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link IOUtil#tryFlush(Flushable...)}
	 */
	@Test
	public
	void
	test_tryFlush()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { null },
			new Object[] { new Flushable[] { new ByteArrayOutputStream(), new ByteArrayOutputStream() } },
			new Object[] { new Flushable[] { null } },
			new Object[] { new Flushable[] { null, null, null } },
			new Object[] { new Flushable[] { null, new ByteArrayOutputStream() } },
			new Object[] { new Flushable[] { new ByteArrayOutputStream(), null } },
			new Object[] { new Flushable[] { new ByteArrayOutputStream(), null } },
			new Object[] { new Flushable[] { new ByteArrayOutputStream(), null, new ByteArrayOutputStream() } },
		};
		
		int iTestCase = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTestCase;
			final Flushable[] flushables = (Flushable[]) TEST_CASE[0];

			try
			{
				IOUtil.tryFlush( flushables );
			}
			catch ( final Throwable e )
			{
				fail( iTestCase + " th test case failed" );
			}
		}
		
		/* Mock Object using Mockito */
		OutputStream out = mock( OutputStream.class );
		IOUtil.tryFlush( out );
		verify( out ).flush();
	}
	

	/**
	 * Test {@link IOUtil#redirect(InputStream, OutputStream)},
	 * {@link IOUtil#redirect(java.io.Reader, StringBuffer)},
	 * {@link IOUtil#redirect(java.io.Reader, StringBuilder)}
	 * and {@link IOUtil#redirect(java.io.Reader, java.io.Writer)}
	 * 
	 * @throws Exception in case of failure in test
	 *
	 * @see IOUtil#redirect(InputStream, OutputStream)
	 * @see IOUtil#redirect(java.io.Reader, StringBuffer)
	 * @see IOUtil#redirect(java.io.Reader, StringBuilder)
	 * @see IOUtil#redirect(java.io.Reader, java.io.Writer)
	 */
	@Test
	public
	void
	test_redirect()
	throws Exception
	{
		final String[] TEST_CASES = new String[] {
			"Hello, World",
		};
		
		int iTestCase = 0;
		for ( final String TEST_CASE : TEST_CASES )
		{
			++iTestCase;

			final ByteArrayInputStream in = new ByteArrayInputStream( TEST_CASE.getBytes() );
			final ByteArrayOutputStream out = new ByteArrayOutputStream();
			IOUtil.redirect( in, out );
			assertArrayEquals( iTestCase + " th test case failed :" + TEST_CASE, TEST_CASE.getBytes(), out.toByteArray() );

			final StringReader reader = new StringReader( TEST_CASE );
			final StringWriter writer = new StringWriter();
			IOUtil.redirect( reader, writer );
			assertEquals( iTestCase + " th test case failed :" + TEST_CASE, TEST_CASE, writer.toString() );

			final StringReader reader2 = new StringReader( TEST_CASE );
			final StringBuilder builder = new StringBuilder();
			IOUtil.redirect( reader2, builder );
			assertEquals( iTestCase + " th test case failed :" + TEST_CASE, TEST_CASE, builder.toString() );

			final StringReader reader3 = new StringReader( TEST_CASE );
			final StringBuffer buffer = new StringBuffer();
			IOUtil.redirect( reader3, buffer );
			assertEquals( iTestCase + " th test case failed :" + TEST_CASE, TEST_CASE, buffer.toString() );

		}
	}
	
	/**
	 * Test {@link IOUtil#getBytes(InputStream)}
	 * 
	 * @throws Exception in case of failure in test
	 *
	 * @see IOUtil#getBytes(InputStream)
	 */
	@Test
	public
	void
	test_getBytes()
	throws Exception
	{
		assertNotNull( IOUtil.getBytes( new ByteArrayInputStream( "Hello".getBytes() ) ) );
		assertTrue( 0 < IOUtil.getBytes( new ByteArrayInputStream( "Hello".getBytes() ) ).length );
	}
	
	
	/**
	 * Test {@link IOUtil#getString(java.io.Reader)},
	 * {@link IOUtil#getString(java.io.Reader, boolean)}
	 * 
	 * @throws Exception in case of failure in test
	 *
	 * @see IOUtil#getString(java.io.Reader)
	 * @see IOUtil#getString(java.io.Reader, boolean)
	 */
	@Test
	public
	void
	test_getString()
	throws Exception
	{
		assertEquals( "", IOUtil.getString( (Reader) null ) );
		assertEquals( "Hello", IOUtil.getString( new StringReader("Hello" ) ) );
		
		assertEquals( "", IOUtil.getString( (StringReader) null, true ) );
		assertEquals( "Hello", IOUtil.getString( new StringReader("Hello" ), true ) );
	}
	
}
