/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.common.util;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

/**
 * StringUtilTest
 * 
 * Test case for {@link StringUtil}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see StringUtil
 */
public class
StringUtilTest
{
    /**
     * @throws Exception in case of failure in test
     *
     * @see StringUtil#nvl(String...)
     */
    @Test
    public
    void
    test_nvl()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "", new String[] { null, null } },
            new Object[] { "Hello", new String[] { "Hello", "World" } },
            new Object[] { "World", new String[] { null, "World" } },
            new Object[] { "hello", new String[] { null, "hello" } },
            new Object[] { "hello", new String[] { "hello", "world" } },
            new Object[] { "hello", new String[] { null, null, "hello", "world" } },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String expected = (String) TEST_CASE[0];
            final String[] input = (String[]) TEST_CASE[1];
            assertEquals( expected, StringUtil.nvl( input ) );
        }
    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#trimLeadingWhitespace(String)}
     */
    @Test
    public void
    test_trimLeadingWhitespace()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "", "" },
            new Object[] { "", "   " },
            new Object[] { null, null },
            new Object[] { "Hello", "  Hello" },
            new Object[] { "H ", " H " },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String expected = (String) TEST_CASE[0];
            final String input = (String) TEST_CASE[1];
            assertEquals( expected, StringUtil.trimLeading( input ) );
        }
    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#trimLeadingWhitespace(String)}
     */
    @Test
    public void
    test_trimTrailingWhitespace()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "", "" },
            new Object[] { "", "   " },
            new Object[] { null, null },
            new Object[] { "Hello", "Hello  " },
            new Object[] { " H", " H " },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String expected = (String) TEST_CASE[0];
            final String input = (String) TEST_CASE[1];
            assertEquals( expected, StringUtil.trimTrailing( input ) );
        }
    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#trimLeadingCharacter(String, char)}
     */
    @Test
    public void
    test_trimLeadingCharacter()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "", "", '-' },
            new Object[] { "   ", "   ", '-' },
            new Object[] { null, null, '-' },
            new Object[] { "Hello  ", "---Hello  ", '-' },
            new Object[] { " H ", " H ", '-' },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String expected = (String) TEST_CASE[0];
            final String input = (String) TEST_CASE[1];
            final char trimChar = (Character) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.trimLeadingCharacter( input, trimChar )
            );
        }
    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#trimTrailingCharacter(String, char)
     */
    @Test
    public void
    test_trimTrailingCharacter()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "", "", '-' },
            new Object[] { "   ", "   ", '-' },
            new Object[] { null, null, '-' },
            new Object[] { "Hello", "Hello---", '-' },
            new Object[] { " H-- ", " H-- ", '-' },
            new Object[] { " H", " H--", '-' },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String expected = (String) TEST_CASE[0];
            final String input = (String) TEST_CASE[1];
            final char trimChar = (Character) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.trimTrailingCharacter( input, trimChar )
            );
        }
    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#trimCharacter(String, char)}
     */
    @Test
    public
    void
    test_trimCharacter()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "", "", '-' },
            new Object[] { "   ", "   ", '-' },
            new Object[] { null, null, '-' },
            new Object[] { "Hello", "Hello---", '-' },
            new Object[] { " H-- ", " H-- ", '-' },
            new Object[] { " H", " H--", '-' },
            new Object[] { " H", "-- H--", '-' },
            new Object[] { "H", "--H--", '-' },
            new Object[] { " - H", "-- - H--", '-' },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String expected = (String) TEST_CASE[0];
            final String input = (String) TEST_CASE[1];
            final char trimChar = (Character) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.trimCharacter( input, trimChar )
            );
        }

    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see StringUtil#getWord(String, int)
     */
    @Test
    public void
    test_getWord()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "I am a boy", 0, "I" },
            new Object[] { "I am a boy", 1, "I" },
            new Object[] { "I am a boy", 2, "am" },
            new Object[] { "I am a boy", 3, "am" },
            new Object[] { "I am a boy", 4, "am" },
            new Object[] { "I am a boy", 5, "a" },
            new Object[] { "I am a boy", 6, "a" },
            new Object[] { "I am a boy", 7, "boy" },
            new Object[] { "I am a boy", 8, "boy" },
            new Object[] { "I am a boy", 9, "boy" },
            new Object[] { "I am a boy", 10, "boy" },
            new Object[] { "I am a boy", 11, "" },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String doc = (String) TEST_CASE[0];
            final int index = (Integer) TEST_CASE[1];
            final String expected = (String) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.getWord( doc, index )
            );
        }
    }

    /**
     * @throws Exception in case of failure in test
     *
     * @see StringUtil#getPreviousWord(String, int)
     */
    @Test
    public void
    test_getPreviousWord()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "I am a boy ", 0, "" },
            new Object[] { "I am a boy ", 1, "I" },
            new Object[] { "I am a boy ", 2, "" },
            new Object[] { "I am a boy ", 3, "a" },
            new Object[] { "I am a boy ", 4, "am" },
            new Object[] { "I am a boy ", 5, "" },
            new Object[] { "I am a boy ", 6, "a" },
            new Object[] { "I am a boy ", 7, "" },
            new Object[] { "I am a boy ", 8, "b" },
            new Object[] { "I am a boy ", 9, "bo" },
            new Object[] { "I am a boy ", 10, "boy" },
            new Object[] { "I am a boy ", 11, "" },
            new Object[] { "I am a boy ", 12, "" },
            new Object[] { "#{\ntest", 2, "#{" },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String doc = (String) TEST_CASE[0];
            final int index = (Integer) TEST_CASE[1];
            final String expected = (String) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.getPreviousWord( doc, index )
            );
        }
    }

    /**
     * Test {@link StringUtil#getLastSegment(String, String)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#getLastSegment(String, String)}
     */
    @Test
    public
    void
    test_getLastSegement()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { null, " ", "" },
            new Object[] { "I am a boy", " ", "boy" },
            new Object[] { "I am a  boy", " ", "boy" },
            new Object[] { "I am a boy ", " ", "" },
            new Object[] { "a,b,c,d", ",", "d" },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String doc = (String) TEST_CASE[0];
            final String separator = (String) TEST_CASE[1];
            final String expected = (String) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.getLastSegment( doc, separator )
            );
        }
    }
    
    /**
     * Test case for {@link StringUtil#getLastStringAfter(String, String)}
     * 
     * @throws Exception
     */
    public void test_getLastStringAfter() throws Exception {
        String[][] testCases = {
                {"ASDF|||JKL", "|||", "JKL"},
                {"THISTHOSETHAT", "H", "AT"},
                {"P|||AAA|||BBB", "|||", "BBB"},
                {"WEF", null, null},
                {"ALL", "", "ALL"},
                {null, "S", null}
        };
        
        for(String[] testCase: testCases) {
            String result = StringUtil.getLastStringAfter(testCase[0], testCase[1]);
            assertEquals(result, testCase[2]);
        }
    }

    /**
     * Test {@link StringUtil#getLastSegment(String, String)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#getLastSegment(String, String)}
     */
    @Test
    public
    void
    test_removeLastSegement()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { null, " ", "" },
            new Object[] { "I am a boy", " ", "I am a" },
            new Object[] { "I am a  boy", " ", "I am a " },        // FIXME ?
            new Object[] { "I am a boy ", " ", "I am a boy" },
            new Object[] { "a,b,c,d", ",", "a,b,c" },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String doc = (String) TEST_CASE[0];
            final String separator = (String) TEST_CASE[1];
            final String expected = (String) TEST_CASE[2];
            assertEquals(
                expected,
                StringUtil.removeLastSegment( doc, separator )
            );
        }
    }

    /**
     * Test {@link StringUtil#split(String, String)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#split(String, String)}
     */
    @Test
    public
    void
    test_split()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "aa.bbb.ccc", ".", new String[] { "aa", "bbb", "ccc" } }
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final String input = (String) TEST_CASE[0];
            final String delimiters = (String) TEST_CASE[1];
            final String[] expected = (String[]) TEST_CASE[2];

            assertArrayEquals( expected, StringUtil.split( input, delimiters ) );
        }
    }

    /**
     * Test {@link StringUtil#enumNameToStringArray(T[] values)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link StringUtil#enumNameToStringArray(T[] values)}
     */
    @Test
    public void test_enumNameToStringArray() throws Exception {
        TestEnumCase [] enums = TestEnumCase.values();
        String [] values = StringUtil.enumNameToStringArray(enums);
        assertTrue(enums.length == values.length);

        StringBuilder enumBuilder = new StringBuilder();
        for (TestEnumCase testEnumCase : enums) {
            enumBuilder.append(testEnumCase.name());
        }

        StringBuilder utilBuilder = new StringBuilder();
        for (String string : values) {
            utilBuilder.append(string);
        }
        assertEquals(enumBuilder.toString(), utilBuilder.toString());
    }

    /**
     * Test {@link StringUtil#removeStart(String, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link StringUtil#removeStart(String, String)}
     */
    @Test
    public void
    test_removeStart()
    throws Exception
    {
        String str = "abcdefg";
        String remove1 = "a";
        String remove2 = "abc";
        String remove3 = "def";

        // null test
        assertEquals( str, StringUtil.removeStart( str, null ) );
        assertEquals( null, StringUtil.removeStart( null, remove1 ) );
        assertEquals( null, StringUtil.removeStart( null, null ) );

        // start with
        assertEquals( "bcdefg", StringUtil.removeStart(str, remove1 ) );
        assertEquals( "defg", StringUtil.removeStart(str, remove2 ) );

        // not start with
        assertEquals( str, StringUtil.removeStart(str, remove3 ) );
    }

    public enum TestEnumCase {
        ABC("ABC"),
        DEF("DEF"),
        GHI("GHI"),
        JKL("JKL");
        private String value;
        TestEnumCase(String v) {
            value = v;
        }
        public String value() {
            return value;
        }
    }
}
