/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.common.util.io;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.tizen.common.util.IOUtil.tryClose;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * BufferTest
 *
 * Test case for {@link Buffer}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see Buffer
 */
public class BufferTest
{

	/**
	 * {@link BufferPool} to use
	 */
	protected BufferPool bufferPool;

	/**
	 * Set up test unit
	 * 
	 * @throws Exception If test unit can't  be set up
	 */
	@Before
	public
	void
	setUp()
	throws Exception
	{
		bufferPool = new BufferPool();

		bufferPool.setSizeOfFragment( 10 );
	}

	/**
	 * Tear down test unit
	 * 
	 * @throws Exception If test unit can't  be tear down
	 */
	@After
	public
	void
	tearDown()
	throws Exception
	{
		bufferPool.close();
	}

	/**
	 * Test {@link Buffer#readInt()}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link Buffer#readInt()}
	 */
	@Test
	public
	void
	test_readInt()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { 135, new byte[] { 0, 0, 0, -121 } }
		};

		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			final int expected = (Integer) TEST_CASE[0];
			final byte[] input = (byte[]) TEST_CASE[1];
			final Buffer buffer = new Buffer( bufferPool );
			try
			{
				for ( int i = 0, n = input.length ; i < n ; ++i )
				{
					buffer.write( input[i] );
				}

				assertEquals( expected, buffer.readInt() );
			}
			finally
			{
				tryClose( buffer );
			}
		}
	}


	/**
	 * Test {@link Buffer#writeInt(int)}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link Buffer#writeInt(int)}
	 */
	@Test
	public
	void
	test_writeInt()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { 135, new int[] { 0, 0, 0, 135 } }
		};

		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			final int input = (Integer) TEST_CASE[0];
			final int[] expected = (int[]) TEST_CASE[1];

			final Buffer buffer = new Buffer( bufferPool );
			try
			{
				buffer.writeInt( input );
				for ( int i = 0, n = expected.length ; i < n ; ++i )
				{
					assertEquals( expected[i], buffer.read() );
				}
			}
			finally
			{
				tryClose( buffer );
			}
		}
	}

	/**
	 * Test {@link Buffer#getBufferSize()}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link Buffer#getBufferSize()}
	 */
	@Test
	public
	void
	test_getBufferSize()
	throws Exception
	{
		final Buffer buffer = new Buffer( bufferPool );
		try
		{
			final BufferOutputStream bufferOut = new BufferOutputStream( buffer );
			bufferOut.write( "12345678901234567890123456789012345678901234567890".getBytes() );

			assertEquals( 5, buffer.getBufferSize() );

			buffer.read();

			assertEquals( 5, buffer.getBufferSize() );

		}
		finally
		{
			tryClose( buffer );
		}

	}

	/**
	 * Test {@link Buffer#reset()}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link Buffer#reset()}
	 */
	@Test
	public
	void
	test_reset()
	throws Exception
	{
		final byte[] TEST = "12345678901234567890123456789012345678901234567890".getBytes();
		final Buffer buffer = new Buffer( bufferPool );

		try
		{
			final BufferOutputStream bufferOut = new BufferOutputStream( buffer );
			bufferOut.write( TEST );

			final BufferInputStream bufferIn = new BufferInputStream( buffer );
			final byte[] result = new byte[TEST.length];

			assertEquals( TEST.length, bufferIn.read( result ) );
			assertArrayEquals( TEST, result );

			buffer.reset();

			assertEquals( TEST.length, bufferIn.read( result ) );
			assertArrayEquals( TEST, result );
		}
		finally
		{
			tryClose( buffer );
		}

	}

	/**
	 * Test {@link Buffer#concatenate(Buffer...)}
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see {@link Buffer#concatenate(Buffer...)}
	 */
	@Test
	public
	void
	test_concatenate()
	throws Exception
	{
		final Buffer buffer1 = new Buffer( bufferPool );
		final Buffer buffer2 = new Buffer( bufferPool );

		final byte[] EXPECTED = "123456789012123456789012".getBytes();
		final byte[] TEST = "123456789012".getBytes();

		try
		{
			final BufferOutputStream bufferOut1 = new BufferOutputStream( buffer1 );
			bufferOut1.write( TEST );

			final BufferOutputStream bufferOut2 = new BufferOutputStream( buffer1 );
			bufferOut2.write( TEST );

			Buffer ret = Buffer.concatenate( buffer1, buffer2 );
			BufferInputStream bufferIn = new BufferInputStream( ret );

			final byte[] result = new byte[2 * TEST.length];

			assertEquals( result.length, ret.size() );
			assertEquals( result.length, bufferIn.read( result ) );
			assertArrayEquals( EXPECTED, result );

		}
		finally
		{
			tryClose( buffer1 );
		}
	}

}
