/*
 *  Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * BonYong Lee <bonyong.lee@samsung.com>
 * Ho Namkoong <ho.namkoong@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */

package org.tizen.common.util.log;

import static junit.framework.Assert.assertTrue;
import static org.powermock.api.mockito.PowerMockito.mock;
import static org.powermock.api.mockito.PowerMockito.when;
import static org.powermock.api.mockito.PowerMockito.whenNew;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;

import org.apache.log4j.Logger;
import org.apache.log4j.spi.LoggingEvent;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.core.classloader.annotations.PowerMockIgnore;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;
import org.slf4j.LoggerFactory;
import org.tizen.common.util.log.UserLogger.Event;
import org.tizen.common.util.log.UserLogger.Page;
import org.tizen.common.util.log.UserLogger.PerformanceInfo;

@RunWith(PowerMockRunner.class)
@PrepareForTest({GoogleAnalyticAppender.class})
@PowerMockIgnore(value = {"org.slf4j.*", "org.apache.log4j.*"})
public class GoogleAnalyticAppenderTest
{
    private org.slf4j.Logger logger = LoggerFactory.getLogger(this.getClass());
    
    /**
     * Sets up the test
     * 
     * @throws Exception
     */
    @Before
    public void setUp() throws Exception {
        final URL u = new URL( GoogleAnalyticAppender.URL_GOOGLE_ANALYTIC );
        HttpURLConnection con = null;
        try {
            con = (HttpURLConnection) u.openConnection();
            con.setDoOutput(true);
            con.getOutputStream();
        } finally {
            if(con != null) {
                con.disconnect();
            }
        }
    }

    /**
     * 
     * TestCase for {@link GoogleAnalyticAppender#append(LoggingEvent)}
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @Test
    public void test_append() throws Exception
    {
        logger.error("Network to the google analytics is not connected");
        URL mockUrl = mock(URL.class);
        HttpURLConnection mockCon = mock(HttpURLConnection.class);
        OutputStream mockOutputStream = mock(OutputStream.class);
        when(mockCon.getOutputStream()).thenReturn(mockOutputStream);
        when(mockCon.getResponseCode()).thenReturn(200);
        when(mockUrl.openConnection()).thenReturn(mockCon);
        whenNew(URL.class).withArguments(GoogleAnalyticAppender.URL_GOOGLE_ANALYTIC).thenReturn(mockUrl);
        OutputStreamWriter mockWriter = mock(OutputStreamWriter.class);
        whenNew(OutputStreamWriter.class).withAnyArguments().thenReturn(mockWriter);
        
        final GoogleAnalyticAppender appender = new GoogleAnalyticAppender(true, true);
        Logger logger = Logger.getLogger(this.getClass());
        LoggingEvent event = new LoggingEvent( null, logger, Level.ERROR, "TI", null );
        
        appender.append(event);
        assertTrue(appender.rc >= 200 && appender.rc < 300);
        
        event = new LoggingEvent(null, logger, Level.PAGE, new Page("Page category"), null);
        appender.append(event);
        assertTrue(appender.rc >= 200 && appender.rc < 300);
        appender.rc = 0;
        
        event = new LoggingEvent(null, logger, Level.EVENT, new Event("Event Category", "Event Action"), null);
        appender.append(event);
        assertTrue(appender.rc >= 200 && appender.rc < 300);
        appender.rc = 0;
        
        long start = System.currentTimeMillis();
        event = new LoggingEvent(null, logger, Level.PERFORM_START, new PerformanceInfo("Performance category", start, 0, "Performance variable"), null);
        appender.append(event);
        assertTrue(appender.rc >= 200 && appender.rc < 300);
        appender.rc = 0;
        
        long end = System.currentTimeMillis();
        event = new LoggingEvent(null, logger, Level.PERFORM_END, new PerformanceInfo("Performance category", start, end, "Performance variable"), null);
        appender.append(event);
        assertTrue(appender.rc >= 200 && appender.rc < 300);
        appender.rc = 0;
        
        try {
            makeException(5);
        }
        catch (IllegalStateException e) {
            event = new LoggingEvent(null, logger, Level.ERROR, "Test Exception", e);
            appender.append(event);
            assertTrue(appender.rc >= 200 && appender.rc < 300);
            appender.rc = 0;
        }
    }

    private void makeException(int i) {
        if(i < 1) {
            throw new IllegalStateException("Test Exception");
        }
        makeException(i-1);
    }
    
    

}