/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Ho Namkoong <ho.namkoong@samsung.com> 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */

package org.tizen.common.sdb;

import static org.junit.Assert.*;
import static org.tizen.common.Shell.*;
import java.io.File;
import java.text.MessageFormat;
import java.util.Random;

import org.tizen.common.IShell;
import org.tizen.common.Pattern;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.StringUtil;
import org.tizen.sdblib.util.Log;

/**
 * <p>
 * SdbTestUtil.
 * 
 * Utils for sdb test
 *
 * </p>
 * 
 * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
 */
class SdbTestUtil {
    
    static final String LOOPBACK_IP = "127.0.0.1";
    static final int MAX_TRIAL = 10;
    static final char CTRL_C = '\u0003';
    static String SDK_PATH;
    static String SDB_PATH;
    
    static {
        SDK_PATH = System.getProperty("TIZEN_SDK");
        
        if(StringUtil.isEmpty(SDK_PATH)) {
            SDB_PATH = "sdb";
        }
        else {
            SDB_PATH = FileUtil.appendPath(SDK_PATH, "tools/sdb");
        }
    }
    
    static final String FILE_NAME_CHAR_WITHOUT_WHITE_SPACE = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890";
    
    /**
     * File name characters for generating random file name
     */
    static final String FILE_NAME_CHAR_LIST = FILE_NAME_CHAR_WITHOUT_WHITE_SPACE + " ";
    
    /**
     * File content characters for generating random file contents
     */
    static final String FILE_CONTENT_CHAR_LIST = FILE_NAME_CHAR_LIST + "\n\t";//`~!@#$%^&*()-=_+[]\\{}|;'\",./<>?";
    
    /**
     * Get ports of connected emulators
     * 
     * @return Emulator ports.
     * @throws Exception
     */
    static int[] getPorts() throws Exception {
        class PortResult {
            @Pattern( pattern = "emulator-{0}\t{1}", index = 0) public String[] devicePortNum;
        };
        
        final PortResult resultString = run( SDB_PATH + " devices" ).expect( new PortResult() );
        
        String[] devicePorts = null;
        
        assertNotNull("result of sdb devices is null", resultString);
        assertNotNull("device port number is null", resultString.devicePortNum);
        devicePorts = resultString.devicePortNum;
        assertTrue("No devices are detected.", resultString.devicePortNum.length > 0);
            
        int resultLength = devicePorts.length;
        final int[] resultPorts = new int[resultLength];
        
        for(int i=0; i < resultLength; i++) {
            resultPorts[i] = Integer.parseInt(devicePorts[i]);
        }
        return resultPorts;
    }
    
    static boolean isNativePluginInstalled() throws Exception {
        if(StringUtil.isEmpty(SDK_PATH)) {
            return false;
        }
        
        String toolChainPath = FileUtil.appendPath(SDK_PATH, "tools/smart-build-interface");
        
        return new File(toolChainPath).exists();
    }
    
    /**
     * Get serial numbers of connected targets
     * 
     * @return Serial numbers of connected targets
     * @throws Exception
     */
    static String[] getSerialNumber() throws Exception {
        
        class SerialResult {
            @Pattern( pattern = "{0}\t{1}", index = 0 ) public String[] deviceSerialName;
        };
        
        String[] result = null;
        final SerialResult resultString = run( SDB_PATH + " devices" ).expect( new SerialResult() );
        
        assertNotNull("result of sdb devices is null", resultString);
        assertNotNull("device serial number is null", resultString.deviceSerialName);
        result = resultString.deviceSerialName;
        assertTrue("No devices are detected.", result != null && result.length > 0);
            
        return result;
    }
    
    /**
     * Generate random string
     * 
     * @param Length String length
     * @param Chars Characters for generating string
     * @return Generated random string
     */
    static String generateRandomString(int length, String chars) {
        char[] resultChars = new char[length];
        
        Random random = new Random();
        int maxIndex = chars.length();
        for(int i=0; i<length; i++) {
            resultChars[i] = chars.charAt(random.nextInt(maxIndex));
        }
        return String.valueOf(resultChars);
    }
    
    /**
     * Generate random file name
     * 
     * @return Random file name
     */
    static String generateRandomFileName() {
        final int maxLen = 14;
        
        Random random = new Random();
        final int fileLen = random.nextInt(maxLen) + 1;
        
        return generateRandomString(fileLen - 1, FILE_NAME_CHAR_LIST) + generateRandomString(1, FILE_NAME_CHAR_WITHOUT_WHITE_SPACE);
    }
    
    /**
     * Generate random directory
     * 
     * @param dir Target directory
     * @param maxDepth Maximum depth of the target directory
     * @param maxFile Maximum file number each directories in target directory can have
     * @param maxDir Maximum directory number each directories in target directory can have
     * @throws Exception
     */
    static void generateRandomDir(File dir, int maxDepth, int maxFile, int maxDir) throws Exception {
        dir.mkdir();
        
        Random random = new Random();
        //at least one file
        int randomFile = random.nextInt(maxFile - 1) + 1;
        for(int i =0; i< randomFile; i++) {
            File childFile = new File(dir, generateRandomFileName());
            while(childFile.exists()) {
                childFile = new File(dir, generateRandomFileName());
            }
            Log.i("FILE CREATING", String.format("file %s is being created", childFile.getAbsolutePath()));
            String content = generateRandomString(random.nextInt(50), FILE_CONTENT_CHAR_LIST);
            FileUtil.createTextFile(childFile, content, null);
        }
        
        if(maxDepth < 1) {
            //do not create more directory
            return;
        }
        
        int randomDir = random.nextInt(maxDir);
        for(int i=0; i<randomDir; i++) {
            File childDir = new File(dir, generateRandomFileName());
            
            while(childDir.exists()) {
                childDir = new File(dir, generateRandomFileName());
            }
            
            Log.i("DIR CREATING", String.format("directory %s is being created", childDir.getAbsolutePath()));
            generateRandomDir(childDir, maxDepth - 1, maxFile, maxDir);
        }
    }
    
    static String[] getWholeString(IShell shell, boolean assertNotNull) throws Exception {
        class WholeResult {
            @Pattern( pattern = "{0}", index = 0) public String wholeStrings[];
        }
        
        WholeResult result = shell.expect(new WholeResult());
        if(assertNotNull) {
            assertNotNull("output of shell is null", result.wholeStrings);
        }
        return result.wholeStrings;
    }
    
    static String[] runAndGetWholeString(String command, boolean assertNotNull, boolean stdError) throws Exception {
        class WholeResult {
            @Pattern( pattern = "{0}", index = 0) public String wholeStrings[];
        }
        
        WholeResult result;
        
        if(!stdError) {
            result = run(command).expect(new WholeResult());
        }
        else {
            result = run(command).expect(new WholeResult(), true);
        }
        
        if(assertNotNull) {
            assertNotNull(MessageFormat.format("Result of command {0} is null", command), result.wholeStrings);
        }
        return result.wholeStrings;
    }
    
    static String[] runAndGetWholeString(String command, boolean assertNotNull) throws Exception {
        return runAndGetWholeString(command, assertNotNull, false);
    }
}
