/*
 * Web IDE - Command Line Interface
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * GyeongSeok Seo <gyeongseok.seo@samsung.com>
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.cli.exec;

import java.text.MessageFormat;

import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.prompter.ChoiceOption;
import org.tizen.common.core.command.prompter.GenericOption;
import org.tizen.common.core.command.prompter.Option;
import org.tizen.common.core.command.sdb.SmartDevelopmentBridgeManager;
import org.tizen.common.util.ArrayUtil;
import org.tizen.common.util.StringUtil;
import org.tizen.sdblib.IDevice;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



/**
 * common class for WRT command
 * 
 * @author GyeongSeok Seo{@literal <gyeongseok.seo@samsung.com>} (S-Core)
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
TargetDeviceSelector
{
    /**
     * <p>
     * logger for this object
     * </p>
     */
    protected final Logger logger = LoggerFactory.getLogger( getClass() );

    protected Prompter prompter;
    protected int sdbtime;

    public int getSdbtime() {
        return this.sdbtime;
    }

    public void setSdbtime(int sdbtime) {
        this.sdbtime = sdbtime;
    }

    public TargetDeviceSelector(Prompter prompter) {
        this.prompter = prompter;
        this.sdbtime = 30000; // default sdb connection wait 30 sec
    }

    protected SmartDevelopmentBridgeManager createSDBManager() {
        return new SmartDevelopmentBridgeManager();
    }

    protected void printDeviceList( IDevice[] devices ) {
        prompter.notify( "-----------------------------" );
        prompter.notify( "Currnet connected target list" );
        prompter.notify( "-----------------------------" );
        for( int i = 0; i < devices.length; i++ ) {
            String msg = MessageFormat.format( "{0}) {1}( {2} )", (i+1), devices[i].toString(), devices[i].getDeviceName() );
            prompter.notify( msg );
        }
    }

    public IDevice[] getConnectDevices() {
        SmartDevelopmentBridgeManager mgr = createSDBManager();

        int waitTime = getSdbtime();
        while ( !mgr.isConnected() ) {
            try {
                Thread.sleep( 100 );
                waitTime -= 100;

                if ( waitTime <= 0 ) {
                    logger.trace( "Sdb demon not connected." );
                    return null;
                }
            } catch (InterruptedException e) {
                prompter.notify( e.getMessage() );
            }
        }

        return mgr.getDevices();
    }

    public boolean isDeviceConnected() {
        return null != ArrayUtil.pickupFirst( getConnectDevices() );
    }

    public boolean isDeviceConnected( String name ) {
        return null != getDevice( name );
    }

    public boolean isDeviceConnected( IDevice device ) {
        if ( device == null ) {
            return false;
        }
        return null != getDevice( device.toString() );
    }

    public IDevice getDevice( String name ) {
        IDevice device = null;
        if ( name == null ) {
            return device;
        }
        IDevice[] devices = getConnectDevices();
        for( int i = 0; i < devices.length; i++ ) {
            if ( name.equals( devices[i].toString() ) || name.equals( devices[i].getDeviceName() ) ) {
                device = devices[i];
                break;
            }
        }
        return device;
    }

    public IDevice selectDevice() {
        IDevice selectDevice = null;
        IDevice[] devices = getConnectDevices();

        if( devices == null ) {
            return selectDevice;
        }

        if ( devices.length == 1 ) {
            selectDevice = ArrayUtil.pickupFirst( devices );
            String msg = MessageFormat.format( "Selected device: {0} ({1})" , selectDevice.toString(), selectDevice.getDeviceName() );
            prompter.notify( msg );
        } else if ( devices.length > 1 ) {
            ChoiceOption yes = new ChoiceOption( "Yes" );
            ChoiceOption no = new ChoiceOption( "No", true );

            final Option option = prompter.interact(
                MessageFormat.format( "Device is not selected.\n" +
                                      "''{0}( {1} )'' device select and\n" +
                                      "Do you want to be continue ?", devices[0].toString(), devices[0].getDeviceName() ),
                yes,
                no
            );

            if ( yes.equals(option) ) {
                selectDevice = ArrayUtil.pickupFirst( devices );
            } else if ( no.equals( option ) ) {
                printDeviceList( devices );

                ChoiceOption cancel = new ChoiceOption( "Cancel", true );
                GenericOption num = new GenericOption( "Num" );

                final Option op = prompter.interact(
                    "Select device or cancel process.",
                    cancel,
                    num
                );

                if( cancel.equals( op ) ) {
                    prompter.notify( "Process is canceled, please selected device." );
                } else {
                    String answer = ((GenericOption)op).getAnswer();
                    if ( !StringUtil.isEmpty( answer ) ) {
                        try {
                            final int index = Integer.parseInt( answer );

                            // answer check
                            if ( 0 < index && index <= devices.length )
                            {
                                selectDevice = ArrayUtil.get( devices, index-1 );
                                String msg = MessageFormat.format( "Select Device is: {0}( {1} )" , selectDevice.toString(), selectDevice.getDeviceName() );
                                prompter.notify( msg );
                            }
                            else
                            {
                                prompter.notify( "Process is canceled, please selected correct device." );
                            }
                        }
                        catch (NumberFormatException e)
                        {
                            prompter.notify( "Process is canceled, please selected correct device. Input is not integer." );
                        }
                    }
                    else
                    {
                        prompter.notify( "Process is canceled, please selected correct device." );
                    }
                }
            }
        }

        return selectDevice;
    }
}

