/*
* Web IDE - Command Line Interface
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.cli.exec.sign;

import static org.tizen.common.util.FilenameUtil.addTailingPath;
import static org.tizen.common.util.IOUtil.tryClose;
import static org.tizen.common.util.ObjectUtil.nvl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import static org.tizen.cli.exec.LaunchOptionConstants.OPT_NOCHECK;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_NOCHECK;
import static org.tizen.cli.exec.LaunchOptionConstants.OPT_DEVELOP;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_DEVELOP_PROFILE;

import org.tizen.cli.exec.AbstractLauncher;
import org.tizen.cli.exec.Help;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.prompter.ChoiceOption;
import org.tizen.common.core.command.prompter.Option;
import org.tizen.common.file.FileHandler;
import org.tizen.common.file.FileHandler.Attribute;
import org.tizen.common.sign.command.ReadSigningProfileFileCommand;
import org.tizen.common.sign.command.SignCommand;
import org.tizen.common.sign.preferences.SigningProfile;
import org.tizen.common.sign.preferences.SigningProfileContainer;
import org.tizen.common.sign.preferences.SigningProfileItem;
import org.tizen.common.util.FilenameUtil;
import org.tizen.sdblib.util.IOUtil;
import org.tizen.sdblib.util.StringUtil;


/**
 * Command Line Interface for creation of signature file
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
Main
extends AbstractLauncher
{
    
    /**
     * <p>
     * Option for profile with/without profiles file
     * 
     * Examples are following
     * <ul>
     * <li>test:/opt/profiles.xml</li>
     * <li>test - equals to test#${HOME}/.tizen/profiles.xml</li>
     * </ul>
     * </p>
     */
    protected static final String OPT_PROFILE = "profile";
    
    /**
     * <p>
     * Description for profile option
     * 
     * This is printed out in usage
     * </p>
     * 
     * @see #OPT_PROFILE
     */
    protected static final String DESC_PROFILE = "Specify profile name with or without profile file. E.g., test or test:/opt/profile.xml.";

    /**
     * <p>
     * separator between profile name and profile file path 
     * </p>
     * 
     * @see #OPT_PROFILE
     */
    protected static final int SEPARATOR_PROFILE = ':';
    
    
    /**
     * <p>
     * JVM property key for default profile path
     * </p>
     */
    protected static final String PROP_CLI_HOME = "cli.home";
    
    /**
     * <p>
     * OS environment variable key for default profile path
     * </p>
     */
    protected static final String ENV_CLI_HOME = "CLI_HOME";
    
    /**
     * <p>
     * default profile file's name
     * </p>
     */
    protected static final String DEFAULT_PROFILE_NAME = "profiles.xml";
    
    /**
     * Entry point for cli main
     * 
     * @param args user input parameter
     * 
     * @throws Exception If unhandled exception occur
     */
    public static
    void
    main(
        final String[] args
    )
    throws Exception
    {
        final Main instance = new Main();
        instance.run( args );
    }
    
    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getOptions()
     */
    @Override
    @SuppressWarnings("static-access")
    protected Options getOptions() {
        final Options opts = super.getOptions();
        
        opts.addOption( OptionBuilder.withLongOpt( OPT_NOCHECK ).withDescription( DESC_NOCHECK ).create( OPT_NOCHECK.substring( 0, 1 ) ) );
//        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_INCLUDE ).withDescription( DESC_INCLUDE ).create( OPT_INCLUDE.substring( 0, 1 ) ) );
//        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_EXCLUDE ).withDescription( DESC_EXCLUDE ).create( OPT_EXCLUDE.substring( 0, 1 ) ) );
        
        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_PROFILE ).withDescription( DESC_PROFILE ).create( OPT_PROFILE.substring( 0, 1 ) ) );
        opts.addOption( OptionBuilder.withLongOpt( OPT_DEVELOP ).withDescription( DESC_DEVELOP_PROFILE ).create( OPT_DEVELOP.substring( 0, 1 ) ) );
        
        return opts;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    @SuppressWarnings("unchecked")
    protected
    void
    execute(
        final CommandLine cmdLine
    )
    throws Exception
    {

        String profileName = cmdLine.getOptionValue( OPT_PROFILE );
        logger.debug( "Profile option :{}", profileName );
        if ( null == profileName )
        {
            throw new ParseException( "profile option is requred" );
        }

        final List<String> args = cmdLine.getArgList();
        logger.trace( "arguments :{}", args );
        
        int nArgs = args.size(); 
        
        String baseDir = convertPath( "." );
        if ( 0 < nArgs )
        {
            baseDir = convertPath( args.get( 0 ) );
        }
        logger.debug( "Base directory :{}", baseDir );

//        final String[] includes = cmdLine.getOptionValues( OPT_INCLUDE );
//        logger.debug( "Includes :{}", CollectionUtil.toString( includes ) );
//        
//        String[] excludes = cmdLine.getOptionValues( OPT_EXCLUDE );
//        if ( null == includes && null == excludes )
//        {
//            excludes = getDefaultExcludes();
//        }
        
        String[] excludes = getDefaultExcludes();
        
        if ( !cmdLine.hasOption( OPT_NOCHECK ) )
        {
            logger.trace( "Check input" );
            if ( !isValidRoot( baseDir ) )
            {
                logger.trace( "Base directory is not valid root" );
                ChoiceOption yes = new ChoiceOption( "Yes" );
                ChoiceOption no = new ChoiceOption( "No", true );
                
                final Prompter prompter = getPrompter();
                
                final Option option = prompter.interact(
                    MessageFormat.format( ".project file doesn''t exist in [{0}]\nDo you want to be continue?", baseDir ),
                    yes,
                    no
                );
                
                if ( no.equals( option ) )
                {
                    prompter.notify( "Process is canceled." );
                    return ;
                }
            }
        }
        
        String profilesFilePath = null;
        
        final int index = profileName.indexOf( SEPARATOR_PROFILE );
        if ( index < 0 )
        {
            profilesFilePath = getDefaultProfilesFilePath();
        }
        else
        {
            profilesFilePath = profileName.substring( index + 1 );
            profileName = profileName.substring( 0, index );
        }
        
        logger.debug( "Profile name :{}", profileName );
        logger.debug( "Profiles file path :{}", profilesFilePath );
        if ( null == profilesFilePath )
        {
            getExecutionContext().getPrompter().error( "Profiles file not found. Try to specify profiles path using -p option or environment variable cli.home" );
            return ;
        }
        profilesFilePath = convertPath( profilesFilePath );
        
        // preload profile XML for convert
        boolean bSuccess = tryConvert( cmdLine, profilesFilePath );
        if ( ! bSuccess ) {
            return;
        }
        
        // read a selected profile from XML
        final ReadSigningProfileFileCommand readProfile = new ReadSigningProfileFileCommand( profilesFilePath, profileName );
        getExecutor().execute( readProfile );
        logger.info( "Profiles file[{}] is read", profilesFilePath );
        
        SigningProfile profile = readProfile.getProfile();
        if ( profile == null )
        {
            getPrompter().error( "No Signing profile( " + profileName + " ) item in " + profilesFilePath );
            return ;
        }
        
        // notify items of selected profile
        for ( int i = 0; i <= SigningProfile.MAX_DISTRIBUTOR; i++ ) {
            SigningProfileItem item = profile.getProfileItem( i );
            if ( ( item != null ) && ( ! StringUtil.isEmpty( item.getKeyLocation() ) ) ) {
                String msg = ( i == SigningProfile.AUTHOR_ORDINAL ) ? "Author certficate: " : "Distributor" + i + " certificate : ";
                getPrompter().notify( msg + item.getKeyLocation() );
            }
        }
        
        if ( cmdLine.hasOption( OPT_DEVELOP ) ) {
            // Always add a developer certificate first because must be "signature1.xml"
            profile.createProfileItemForDeveloper();
        }
        
        // sign
        final SignCommand command = new SignCommand( baseDir, profile );
        command.setIncludes( null );
        command.setExcludes( excludes );
        getExecutor().execute( command );
    }

    protected boolean tryConvert(final CommandLine cmdLine, String profilesFilePath) throws IOException {
        SigningProfileContainer container = null;
        
        // preload profile XML for convert
        if ( cmdLine.hasOption( OPT_DEVELOP ) ) {
            container = new SigningProfileContainer();
        } else {
            container = new SigningProfileContainer() {
                @Override
                protected void preprocessForItemLoad(SigningProfile profile,
                        String version) {
                    // remove automatically insert dist1 for CLI
                }
            };
        }
        
        InputStream in = getFileHandler().read( profilesFilePath );
        String version = null;
        
        try {
            version = container.readProfileXML( in );
            logger.info( "loaded XML version: {}", version );
            if ( version == null ) {
                getPrompter().error( "Cannot load profiles in " + profilesFilePath );
                return false;
            }
        } finally {
            IOUtil.tryClose( in );
        }
        
        // if old version, rewrite to new version.
        if ( StringUtil.isEmpty( version ) ) {
            getPrompter().notify( "Old version profile detected. Converting this automatically." );
            
            ByteArrayOutputStream outStream = new ByteArrayOutputStream();
            container.writeProfileXML( outStream );
            InputStream inForOut = new ByteArrayInputStream( outStream.toByteArray() );
            
            try {
                getFileHandler().write( profilesFilePath, inForOut );
            } finally {
                tryClose( inForOut );
            }
        }
        
        return true;
    }
    
    /**
     * Return default profiles.xml's path
     * 
     * default profile path is specified by jvm propertyes naming "cli.home"
     * 
     * or os environment variable "CLI_HOME"
     * 
     * @return default profiles path
     */
    protected
    String
    getDefaultProfilesFilePath()
    {
        
        final String cliHome =
        nvl( System.getProperty( PROP_CLI_HOME ), System.getenv( ENV_CLI_HOME ) );
        
        if ( null == cliHome )
        {
            return null;
        }
        
        return FilenameUtil.addTailingPath( cliHome, "conf/" + DEFAULT_PROFILE_NAME );
    }

    /**
     * Check if <code>path</code> is Tizen web project root
     * 
     * @param path directory path to check
     * 
     * @return <code>true</code> if <code>path</code> is Tize web project root
     * 
     * @throws IOException If <code>path</code>'s sub files can't be accessed
     */
    protected
    boolean
    isValidRoot(
        final String path
    )
    throws IOException
    {
        final FileHandler fileHandler = getFileHandler();
        if ( !fileHandler.is( addTailingPath( path, ".project" ), Attribute.EXISTS ) )
        {
            return false;
        }
        
        return true;
    }

    /**
     * Return default excludes file patterns
     * 
     * @return default excludes file patterns
     */
    protected
    String[] getDefaultExcludes()
    {
        return new String[] { ".*", "signature*.xml" };
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#createHelp()
     */
    @Override
    protected Help createHelp() {
        Help help = super.createHelp();

        help.setSyntax( help.getSyntax() + getSyntax() );
        return help;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getSyntax()
     */
    @Override
    protected
    String
    getSyntax()
    {
        return  " [options]";
    }

}
