/*
* Native IDE - Command Line Interface
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* Ho Namkoong <ho.namkoong@samsung.com>
* BonYong Lee <bonyong.lee@samsung.com>
* Kangho Kim <kh5325.kim@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/

package org.tizen.cli.exec.gen;

import java.io.File;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.tizen.cli.exec.AbstractLauncher;
import org.tizen.cli.exec.Help;
import org.tizen.cli.exec.pack.CLIMultiPackager;
import org.tizen.common.TizenPlatformConstants;
import org.tizen.common.core.application.InstallPathConfig;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.prompter.ChoiceOption;
import org.tizen.common.core.command.prompter.Option;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.StringUtil;
import org.tizen.nativecommon.templateengine.TizenTemplateEngine;
import org.tizen.nativecommon.templateengine.build.BuildPluginManagerForCLI;
import org.tizen.nativecommon.templateengine.build.model.BuildProperty;
import org.tizen.nativecommon.templateengine.build.model.DevPackageModel;
import org.tizen.nativecommon.templateengine.build.model.RootstrapModel;
import org.tizen.nativecommon.templateengine.build.model.TargetModel;
import org.tizen.nativecommon.templateengine.build.util.BuildPluginUtil;
import org.tizen.nativecommon.templateengine.model.TizenTemplate;
import org.tizen.nativecpp.misc.core.NewAppXmlStore;

public class NativeMain extends AbstractLauncher {
    
    private static final String PROJECTTYPE_APP = "app";
    private static final String PROJECTTYPE_STATIC = "static";
    private static final String PROJECTTYPE_SHARED = "shared";
    
    private static final String SAMPLE_EMPTY = "empty";
    private static final String SAMPLE_FORM = "form";
    private static final String SAMPLE_SERVICE = "service";
    private static final String SAMPLE_SHARED = "shared";
    private static final String SAMPLE_STATIC = "static";
    private static final String SAMPLE_TAB = "tab";
    
    private static final String OPTION_PROJECTPATH = "path";
    private static final String OPTION_PROJECTNAME = "name";
    private static final String OPTION_SAMPLE = "sample";
    private static final String OPTION_FORCE = "force";
    
    private static final String OPTION_MAKE_PROJECTTYPE = "type";
    private static final String OPTION_MAKE_BINARY = "binary";
    private static final String OPTION_CMD_PROJECT = "project";
    private static final String OPTION_CMD_MAKEFILE = "makefile";
    
    private static final String DESC_PROJECTPATH = Messages.getString("NativeMain.DESC_PROJECTPATH"); //$NON-NLS-1$
    private static final String DESC_PROJECTNAME = Messages.getString("NativeMain.DESC_PROJECTNAME"); //$NON-NLS-1$
    private static final String DESC_SAMPLE = MessageFormat.format(Messages.getString("NativeMain.DESC_SAMPLE"), 
            SAMPLE_EMPTY, SAMPLE_FORM, SAMPLE_SERVICE, SAMPLE_SHARED, SAMPLE_STATIC, SAMPLE_TAB); //$NON-NLS-1$
    private static final String DESC_CMD_PROJECT = Messages.getString("NativeMain.DESC_CMD_PROJECT"); //$NON-NLS-1$
    private static final String DESC_CMD_MAKEFILE = Messages.getString("NativeMain.DESC_CMD_MAKEFILE"); //$NON-NLS-1$
    private static final String DESC_FORCE = Messages.getString("NativeMain.DESC_FORCE"); //$NON-NLS-1$
    
    private static final String DESC_MAKE_PROJECTTYPE = MessageFormat.format(Messages.getString("NativeMain.DESC_MAKE_PROJECTTYPE"), PROJECTTYPE_APP, PROJECTTYPE_STATIC, PROJECTTYPE_SHARED); //$NON-NLS-1$
    private static final String DESC_MAKE_BINARY = Messages.getString("NativeMain.DESC_MAKE_BINARY"); //$NON-NLS-1$
    
    private static final String TEMPLATE_DIRECTORY = "native/cpp/Template/Tizen Native";
    private static final String SAMPLE_DIR_EMPTYPROJECT = TEMPLATE_DIRECTORY + "/Empty Application/empty";
    private static final String SAMPLE_DIR_FORM = TEMPLATE_DIRECTORY + "/Form-based Application/formapp";
    private static final String SAMPLE_DIR_SHARED = TEMPLATE_DIRECTORY + "/Library/sharedlibrary";
    private static final String SAMPLE_DIR_STATIC = TEMPLATE_DIRECTORY + "/Library/staticlibrary";
    private static final String SAMPLE_DIR_SERVICE = TEMPLATE_DIRECTORY + "/Service Application/serviceapp";
    private static final String SAMPLE_DIR_TAB = TEMPLATE_DIRECTORY + "/Tab-based Application/tabapp";
    
    private static final HashMap<String, String> SAMPLE_DIR_MAP = new HashMap<String, String>();
    private static final HashMap<String, String> PROJECTTYPE_MAP = new HashMap<String, String>();

    
    static {
        SAMPLE_DIR_MAP.put(SAMPLE_EMPTY, SAMPLE_DIR_EMPTYPROJECT);
        SAMPLE_DIR_MAP.put(SAMPLE_FORM, SAMPLE_DIR_FORM);
        SAMPLE_DIR_MAP.put(SAMPLE_SERVICE, SAMPLE_DIR_SERVICE);
        SAMPLE_DIR_MAP.put(SAMPLE_SHARED, SAMPLE_DIR_SHARED);
        SAMPLE_DIR_MAP.put(SAMPLE_STATIC, SAMPLE_DIR_STATIC);
        SAMPLE_DIR_MAP.put(SAMPLE_TAB, SAMPLE_DIR_TAB);
        
        PROJECTTYPE_MAP.put(PROJECTTYPE_APP, TizenTemplateEngine.PROJECTTYPE_APP);
        PROJECTTYPE_MAP.put(PROJECTTYPE_SHARED, TizenTemplateEngine.PROJECTTYPE_SHARED);
        PROJECTTYPE_MAP.put(PROJECTTYPE_STATIC, TizenTemplateEngine.PROJECTTYPE_STATIC);
    }
    
    /**
     * Entry point for cli main
     * 
     * @param args user input parameter
     * 
     * @throws Exception If unhandled exception occur
     */
    public static void main(String[] args) throws Exception {
        
        final NativeMain instance = new NativeMain();
        
        instance.run( args );
    }

    @Override
    protected void execute(CommandLine cmdLine) throws Exception {
        
        if(getInputCmd().equals(OPTION_CMD_MAKEFILE)) {
            executeMakefile(cmdLine);
        }
        else {
            executeProject(cmdLine);
        }
    }
    
    private void executeProject(CommandLine cmdLine) throws Exception {
        final Prompter prompter = getPrompter();
        logger.trace("get user input");
        
        String sampleName = cmdLine.getOptionValue(OPTION_SAMPLE);
        
        String samplePath = InstallPathConfig.getSamplesPath();
        String sampleDir = SAMPLE_DIR_MAP.get(sampleName);
        
        if(null == sampleDir) {
            prompter.error(MessageFormat.format(Messages.getString("NativeMain.EXCEPTION_SAMPLE_NOT_SUPPORT"), sampleName) );
            return;
        }
        
        sampleDir = FileUtil.appendPath(samplePath, sampleDir);
        
        String projectName = cmdLine.getOptionValue(OPTION_PROJECTNAME);
        if(StringUtil.isEmpty(projectName)) {
            projectName = sampleName;
        }
        
        String targetDir = cmdLine.getOptionValue(OPTION_PROJECTPATH);
        if(StringUtil.isEmpty(targetDir)) {
            File currentFile = new File("");
            targetDir = FileUtil.appendPath(currentFile.getAbsolutePath(), projectName);
        }
        
        if(new File(targetDir).exists()) {
            
            if(!cmdLine.hasOption(OPTION_FORCE)) {
                ChoiceOption yes = new ChoiceOption( "Yes" );
                ChoiceOption no = new ChoiceOption( "No", true );
                
                final Option option = prompter.interact(
                        MessageFormat.format( Messages.getString("NativeMain.DIR_EXIST"), targetDir ), //$NON-NLS-1$
                        yes,
                        no
                    );
                
                if ( no.equals( option ) )
                {
                    prompter.notify( Messages.getString("NativeMain.PROCESS_CANCEL") ); //$NON-NLS-1$
                    return ;
                }
            }
        }
        
        prompter.notify(MessageFormat.format(Messages.getString("NativeMain.CREATE_A_PROJECT"), projectName)); //$NON-NLS-1$
        TizenTemplate template = new TizenTemplate(new File(sampleDir).getAbsolutePath(), projectName, new File(targetDir).getAbsolutePath());
        TizenTemplateEngine.createProject(template, new CLIProjectGenerator());
        prompter.notify(MessageFormat.format(Messages.getString("NativeMain.CREATE_SUCCESS"), 
                template.getProjectPath(), template.getProjectPath() + File.separatorChar + CLIMultiPackager.BIN_FOLDER_NAME));
    }

    private void executeMakefile(CommandLine cmdLine) throws Exception {
        
        Prompter prompter = getPrompter();
        
        String projectType = cmdLine.getOptionValue(OPTION_MAKE_PROJECTTYPE);
        String projectId = PROJECTTYPE_MAP.get(projectType);
        
        if(StringUtil.isEmpty(projectId)) {
            prompter.error(MessageFormat.format(Messages.getString("NativeMain.EXCEPTION_WRONG_PROJECT_TYPE"), projectType) );
            return;
        }
        
        String binaryName = cmdLine.getOptionValue(OPTION_MAKE_BINARY);
        String projectPath = new File(StringUtil.EMPTY_STRING).getAbsolutePath();
        
        if(StringUtil.isEmpty(binaryName)) {
            //Only application has a manifest.xml
            if(projectType.equals(PROJECTTYPE_APP)) {
                File manifestFile = new File(TizenPlatformConstants.MANIFEST_XML_FILE_NAME);
                if(!manifestFile.exists()) {
                    prompter.error("manifext.xml does not exist.");
                    return;
                }
                NewAppXmlStore store = new NewAppXmlStore();
                store.setManifestFile(manifestFile.getAbsolutePath());
                logger.trace("loading manifest from " + manifestFile.getAbsolutePath());
                store.loadXml();
                binaryName = store.getModel().getAppList().get(0).getName();
                logger.trace("loaded (binaryName - " + binaryName + ")");
            }
            else {
                binaryName = FileUtil.getFileNameFromPath(projectPath);
            }
        }
        
        TargetModel target = BuildPluginManagerForCLI.getDefaultTarget(projectId, BuildPluginUtil.ARCH_I386);
        RootstrapModel rootstrap = target.getRootstrap();
        HashMap<String, DevPackageModel> devPkgMap = BuildPluginManagerForCLI.DEVPKG_MAP.get(rootstrap.getId());
        
        ArrayList<DevPackageModel> devPkgList = new ArrayList<DevPackageModel>();
        for(DevPackageModel devPkg: devPkgMap.values()) {
            devPkgList.add(devPkg);
        }
        
        TizenTemplate template = new TizenTemplate("", binaryName, projectPath);
        Map<String, String> valueStore = new HashMap<String, String>();
        
        template.setProperty(TizenTemplate.PROPERTY_KEY_PROJECTTYPE, projectId);
        template.setProperty(TizenTemplate.PROPERTY_KEY_EMUL_TARGET, target);
        
        CLIProjectGenerator generator = new CLIProjectGenerator();
        generator.setTemplate(template);
        BuildProperty buildProperty = BuildPluginManagerForCLI.getBuildPropertyFromDevPkgs(devPkgList, template, target, new CLIProjectGenerator());
        
        template.setProperty(TizenTemplate.PROPERTY_KEY_EMUL_BUILD_PROPERTY, buildProperty);
        
        generator.doRemainingJob(devPkgList, template, target);
        
        generator.pressDefaultPostValueStore(valueStore, template);
        generator.pressPostValueStoreFromDevPkg(valueStore, template);
        generator.createBuildFiles(valueStore, template);
        
        prompter.notify(MessageFormat.format(Messages.getString("NativeMain.MAKE_CREATE_SUCCESS"), 
                projectPath + File.separatorChar + CLIMultiPackager.BIN_FOLDER_NAME));
    }
    
    @SuppressWarnings("static-access")
    @Override
    protected Options getCommands() {
        Options result = super.getCommands();
        
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_CMD_PROJECT).hasArg(false).withDescription(DESC_CMD_PROJECT).create());
        result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_CMD_MAKEFILE).hasArg(false).withDescription(DESC_CMD_MAKEFILE).create());
        
        return result;
    }
    
    @SuppressWarnings("static-access")
    @Override
    protected Options getOptions() {
        Options result = super.getOptions();
        
        if(getInputCmd().equals(OPTION_CMD_MAKEFILE)){
            result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_MAKE_PROJECTTYPE).withDescription(DESC_MAKE_PROJECTTYPE).isRequired().create("t"));
            result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_MAKE_BINARY).withDescription(DESC_MAKE_BINARY).create("b"));
        }
        else {
            result.addOption(OptionBuilder.withLongOpt(OPTION_FORCE).withDescription(DESC_FORCE).create("f"));
            result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_SAMPLE).withDescription(DESC_SAMPLE).isRequired().create("s"));
            result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_PROJECTPATH).withDescription(DESC_PROJECTPATH).create("p"));
            result.addOption(OptionBuilder.hasArg().withLongOpt(OPTION_PROJECTNAME).withDescription(DESC_PROJECTNAME).create("n"));
        }
        return result;
    }
    
    @Override
    protected boolean isPrintHelp() {
        return true;
    }
    
    @Override
    protected String getUsageDescription() {
        return "Create a native project or a makefile";
    }
    
    private Options getSuperOptions() {
        return super.getOptions();
    }
    
    @Override
    protected Help createHelp() {
        return new Help(getOptions(), getCommands(), getInputCmd(), getUsageDescription()) {
            
            NativeMain outer = NativeMain.this;
            private Help defaultHelp = new Help(outer.getSuperOptions(), outer.getCommands(), outer.getInputCmd(), outer.getUsageDescription());
            
            @Override
            public String getHelp() {
                Options cmds = this.getCmds();
                if(cmds.getOption(this.getInputCmd()) == null) {
                    return defaultHelp.getHelp();
                }
                return super.getHelp();
            }
            
        };
    }
}