/*
* Web IDE - Command Line Interface
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* GyeongSeok Seo <gyeongseok.seo@samsung.com>
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.cli.exec.install;

import static org.tizen.cli.exec.LaunchOptionConstants.DESC_FAILED_CANNOT_FIND_CONNECTED_DEVICE;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_FAILED_CANNOT_FIND_SELECTED_DEVICE;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_WIDGETFILE;
import static org.tizen.cli.exec.LaunchOptionConstants.DESC_WIDGETID;
import static org.tizen.cli.exec.LaunchOptionConstants.OPT_WIDGETFILE;
import static org.tizen.cli.exec.LaunchOptionConstants.OPT_WIDGETID;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.tizen.cli.exec.Help;
import org.tizen.cli.exec.WRTLauncher;
import org.tizen.common.core.command.Executor;
import org.tizen.common.core.command.Prompter;
import org.tizen.common.core.command.sdb.PushSdbCommand;
import org.tizen.common.file.FileHandler;
import org.tizen.common.file.FileHandler.Attribute;
import org.tizen.common.sdb.command.message.ErrorMessageType;
import org.tizen.common.util.FilenameUtil;
import org.tizen.common.util.StringUtil;
import org.tizen.web.zimlaunch.cli.command.CheckInstallCommand;
import org.tizen.web.zimlaunch.cli.command.InstallCommand;
import org.tizen.web.zimlaunch.cli.command.KillCommand;

/**
 * Command Line Interface for install widget in target
 * 
 * @author GyeongSeok Seo{@literal <gyeongseok.seo@samsung.com>} (S-Core)
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
Main
extends WRTLauncher
{
    protected String FAILED_INSTALL = "Failed to install: ";

    /**
     * <p>
     *Description for failed operation
     * 
     * This is printed when device is not connected
     * </p>
     */
    String DESC_FAILED_CANNOT_FIND_SELECTED_WIDGET = "can't find selected widget file";

    /**
     * Entry point for cli main
     * 
     * @param args user input parameter
     * 
     * @throws Exception If unhandled exception occur
     */
    public static
    void
    main(
        final String[] args
    )
    throws Exception
    {
        final Main instance = new Main();
        instance.run( args );
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#execute(org.apache.commons.cli.CommandLine)
     */
    @Override
    protected
    void
    execute(
        final CommandLine cmdLine
    )
    throws Exception
    {
        super.execute( cmdLine );

        final Prompter prompter = getPrompter();
        final FileHandler fileHandler = getFileHandler();
        final Executor exec = getExecutor();

        // local widget file path control
        String inputName = cmdLine.getOptionValue( OPT_WIDGETFILE );
        logger.trace("input file name : {} ", inputName);
        if ( StringUtil.isEmpty( inputName ) ) {
            prompter.notify( FAILED_INSTALL + DESC_FAILED_CANNOT_FIND_SELECTED_WIDGET );
            exit( 1 );
        }
        String widgetId = cmdLine.getOptionValue( OPT_WIDGETID );

        String filePath = (String)fileHandler.get( inputName, Attribute.PATH );
        logger.trace( "local widget file path : {}", filePath );

        String fileName = FilenameUtil.getFilename( inputName );
        logger.trace( "local widget file name : {}", fileName );

        // core operation
        InstallCommand install_command = new InstallCommand(getDevice(), fileName);
        install_command.setLocalPath( filePath );
        install_command.setDevice( getDevice() );
        install_command.setTime( getTime() );
        String remotePath = install_command.getRemotePath();
        logger.trace( "remote widget file path : {}", remotePath );

        PushSdbCommand push_command = new PushSdbCommand( filePath, remotePath );
        push_command.setDevice( getDevice() );

        // widget push
        logger.trace( "push command");
        exec.execute( push_command );
        if ( "Failed".equals( push_command.getResult() ) ) {
            exit ( 1 );
        }

        if ( !StringUtil.isEmpty( widgetId ) ) {
            logger.trace( "same widget's pre-install check and kill process" );
            CheckInstallCommand check_install_cmd = new CheckInstallCommand( widgetId );
            check_install_cmd.setDevice( getDevice() );
            check_install_cmd.setTime( getTime() );
            check_install_cmd.setPrintOption( false );
            exec.execute( check_install_cmd );
            if ( check_install_cmd.isInstalled() ) {
                KillCommand kill_cmd = new KillCommand( widgetId );
                kill_cmd.setPrintOption( false );
                kill_cmd.setDevice( getDevice() );
                kill_cmd.setTime( getTime() );
                exec.execute( kill_cmd );
            }
        }

        // widget install
        logger.trace( "install command");
        exec.execute( install_command );

        String[] result = install_command.getResult().split("\\s+");
        int returnCode = Integer.parseInt(result[result.length-1]);
        if (returnCode != 0) {
            ErrorMessageType type = ErrorMessageType.getErrorType(returnCode);
            prompter.notify(type.getManagement());
            exit(returnCode);
        }
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getOptions()
     */
    @Override
    @SuppressWarnings("static-access")
    protected
    Options
    getOptions()
    {
        final Options opts = super.getOptions();
        opts.addOption( OptionBuilder.hasArg().isRequired().withLongOpt( OPT_WIDGETFILE ).withDescription( DESC_WIDGETFILE ).create( OPT_WIDGETFILE.substring( 0, 1 ) ) );
        opts.addOption( OptionBuilder.hasArg().withLongOpt( OPT_WIDGETID ).withDescription( DESC_WIDGETID ).create( OPT_WIDGETID.substring( 0, 1 ) ) );

        return opts;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.WRTLauncher#getDeviceNotConnectedMessage()
     */
    @Override
    protected String getDeviceNotConnectedMessage() {
        return FAILED_INSTALL + DESC_FAILED_CANNOT_FIND_CONNECTED_DEVICE;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.WRTLauncher#getDeviceNotSelectedMessage()
     */
    @Override
    protected String getDeviceNotSelectedMessage() {
        return FAILED_INSTALL + DESC_FAILED_CANNOT_FIND_SELECTED_DEVICE;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#createHelp()
     */
    @Override
    protected Help createHelp() {
        Help help = super.createHelp();

        help.setSyntax( help.getSyntax() + getSyntax() );
        return help;
    }

    /* (non-Javadoc)
     * @see org.tizen.cli.exec.AbstractLauncher#getSyntax()
     */
    @Override
    protected
    String
    getSyntax()
    {
        return " [options] --widget widget-file-name.wgt";
    }
}
