/*
 * Copyright (c) 2018 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#ifndef __TIZEN_SYSTEM_UPDATE_CONTROL_H__
#define __TIZEN_SYSTEM_UPDATE_CONTROL_H__

#include <tizen.h>
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup CAPI_SYSTEM_UPDATE_CONTROL_MODULE
 * @{
 */


/**
 * @brief Enumeration for the update control error.
 * @since_tizen 5.0
 */
typedef enum {
	UPDATE_CONTROL_ERROR_NONE = TIZEN_ERROR_NONE, /**< Successful */
	UPDATE_CONTROL_ERROR_INVALID_PARAMETER = TIZEN_ERROR_INVALID_PARAMETER, /**< Invalid parameter */
	UPDATE_CONTROL_ERROR_OUT_OF_MEMORY = TIZEN_ERROR_OUT_OF_MEMORY, /**< Out of memory */
	UPDATE_CONTROL_ERROR_FILE_NO_SPACE_ON_DEVICE = TIZEN_ERROR_FILE_NO_SPACE_ON_DEVICE, /**< No space left on device */
	UPDATE_CONTROL_ERROR_KEY_NOT_FOUND = TIZEN_ERROR_KEY_NOT_AVAILABLE, /**< Specified key is not found */
	UPDATE_CONTROL_ERROR_KEY_REJECTED = TIZEN_ERROR_KEY_REJECTED, /**< Key is not available */
	UPDATE_CONTROL_ERROR_NOT_SUPPORTED = TIZEN_ERROR_NOT_SUPPORTED, /**< Not supported */
	UPDATE_CONTROL_ERROR_PERMISSION_DENIED = TIZEN_ERROR_PERMISSION_DENIED, /**< Permission denied */
	UPDATE_CONTROL_ERROR_CONNECTION_ABORTED = TIZEN_ERROR_CONNECTION_ABORTED, /**< Software caused connection abort */
	UPDATE_CONTROL_ERROR_CONNECTION_REFUSED = TIZEN_ERROR_CONNECTION_REFUSED, /**< Connection refused */
	UPDATE_CONTROL_ERROR_PROTOCOL_NOT_SUPPORTED = TIZEN_ERROR_PROTOCOL_NOT_SUPPORTED, /**< Protocol not supported */
	UPDATE_CONTROL_ERROR_TIMED_OUT = TIZEN_ERROR_TIMED_OUT, /**< Time out */
	UPDATE_CONTROL_ERROR_INVALID_OPERATION = TIZEN_ERROR_INVALID_OPERATION, /**< Function not implemented */
	UPDATE_CONTROL_ERROR_INVALID_PACKAGE = TIZEN_ERROR_UPDATE_CONTROL | 0x01, /**< Invalid package */
	UPDATE_CONTROL_ERROR_INVALID_URI = TIZEN_ERROR_UPDATE_CONTROL | 0x02, /**< Invalid URI */
	UPDATE_CONTROL_ERROR_PACKAGE_NOT_SUPPORTED = TIZEN_ERROR_UPDATE_CONTROL | 0x03, /**< Package type not supported */
	UPDATE_CONTROL_ERROR_SYSTEM_ERROR = TIZEN_ERROR_UPDATE_CONTROL | 0x04, /**< System error */
} update_control_error_e;


/**
 * @brief Enumeration for the update control properties.
 * @since_tizen 5.0
 */
typedef enum {
	UPDATE_CONTROL_PROPERTY_NEW_VERSION = 0, /**< New version */
	UPDATE_CONTROL_PROPERTY_PACKAGE_URI, /**< URI for update package */
	UPDATE_CONTROL_PROPERTY_RESULT, /**< Update result */
	UPDATE_CONTROL_PROPERTY_PACKAGE_SIZE, /**< Size of update package */
	UPDATE_CONTROL_PROPERTY_DESCRIPTION, /**< Update description */
	UPDATE_CONTROL_PROPERTY_UPDATE_AVAILABLE, /**< Updatability flag */
} update_control_property_e;


/**
 * @brief Initializes the update controller.
 *
 * @since_tizen 5.0
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 * @see update_control_deinitialize()
 */
int update_control_initialize(void);


/**
 * @brief Deinitializes the update controller.
 *
 * @since_tizen 5.0
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 * @see update_control_initialize()
 */
int update_control_deinitialize(void);


/**
 * @brief Requests checking new version of firmware.
 *
 * @since_tizen 5.0
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_CONNECTION_REFUSED Connection refused
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 */
int update_control_check_new_version(void);


/**
 * @brief Requests downloading new version of firmware.
 *
 * @since_tizen 5.0
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_FILE_NO_SPACE_ON_DEVICE No space left on device
 * @retval #UPDATE_CONTROL_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UPDATE_CONTROL_ERROR_CONNECTION_REFUSED Connection refused
 * @retval #UPDATE_CONTROL_ERROR_CONNECTION_ABORTED Software caused connection abort
 * @retval #UPDATE_CONTROL_ERROR_PROTOCOL_NOT_SUPPORTED Protocol not supported
 * @retval #UPDATE_CONTROL_ERROR_INVALID_URI Invalid URI
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 */
int update_control_download_package(void);


/**
 * @platform
 * @brief Requests triggering update to new firmware.
 *
 * @since_tizen 5.0
 * @privlevel platform
 * @privilege %http://tizen.org/privilege/updatecontrol.admin
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_PERMISSION_DENIED Permission denied
 * @retval #UPDATE_CONTROL_ERROR_TIMED_OUT Time out
 * @retval #UPDATE_CONTROL_ERROR_INVALID_PACKAGE Invalid package
 * @retval #UPDATE_CONTROL_ERROR_PACKAGE_NOT_SUPPORTED Package type not supported
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 */
int update_control_do_update(void);


/**
 * @platform
 * @brief Makes reservation for update.
 * @details Makes update reservation at a specific time. At the given time, it automatically checks, downloads and updates to a new version. To cancel the reservation, call update_control_cancel_reservation().
 *
 * @since_tizen 5.0
 * @privlevel platform
 * @privilege %http://tizen.org/privilege/updatecontrol.admin
 * @remarks Only one reservation can be used at any given time. If a new reservation is made, the previous one will be removed.
 * @param[in] reservation_time The time to start update
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UPDATE_CONTROL_ERROR_PERMISSION_DENIED Permission denied
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 */
int update_control_make_reservation(struct tm *reservation_time);


/**
 * @platform
 * @brief Cancels the update reservation.
 *
 * @since_tizen 5.0
 * @privlevel platform
 * @privilege %http://tizen.org/privilege/updatecontrol.admin
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_PERMISSION_DENIED Permission denied
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 */
int update_control_cancel_reservation(void);


/**
 * @brief Gets the update property of the given key.
 *
 * @since_tizen 5.0
 * @remarks The @a value must be released using free()
 * @param[in] property The key of property to get
 * @param[out] value The value of given property
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #UPDATE_CONTROL_ERROR_NONE Successful
 * @retval #UPDATE_CONTROL_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #UPDATE_CONTROL_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #UPDATE_CONTROL_ERROR_KEY_NOT_FOUND Specified property not found
 * @retval #UPDATE_CONTROL_ERROR_KEY_REJECTED Specified property not available
 * @retval #UPDATE_CONTROL_ERROR_SYSTEM_ERROR System error
 * @retval #UPDATE_CONTROL_ERROR_INVALID_OPERATION Function not implemented
 * @retval #UPDATE_CONTROL_ERROR_NOT_SUPPORTED Not supported
 */
int update_control_get_property(update_control_property_e property, void **value);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* __TIZEN_SYSTEM_UPDATE_CONTROL_H__ */
