/*
 * Copyright (c) 2000 - 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 *
 *
 * @file ckmc-manager.h
 * @version 1.0
 * @brief Provides management functions(storing, retrieving, and removing) for keys,
 *        certificates and data of a user and additional crypto functions.
 */


#ifndef __TIZEN_CORE_CKMC_MANAGER_H
#define __TIZEN_CORE_CKMC_MANAGER_H


#include <stddef.h>
#include <sys/types.h>
#include <tizen.h>
#include <ckmc/ckmc-type.h>
#include <ckmc/ckmc-error.h>


#ifdef __cplusplus
extern "C" {
#endif


/**
 * @addtogroup CAPI_KEY_MANAGER_CLIENT_MODULE
 * @{
 */


/**
 * @brief Stores a key inside key manager based on the provided policy.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks Currently API supports seven types of keys. These are RSA public/private key, DSA public/private key, ECDSA public/private key, and AES symmetric key.
 * @remarks key_type in key may be set to #CKMC_KEY_NONE as an input. key_type is determined inside key manager during storing keys.
 * @remarks Some private key files are protected by a password. If raw_key in key read from those encrypted files is encrypted with a password, the password should be provided in the #ckmc_key_s structure.
 * @remarks If password in policy is provided, the key is additionally encrypted with the password in the policy.
 * @param[in] alias The name of a key to be stored
 * @param[in] key The key's binary value to be stored
 * @param[in] policy The policy about how to store a key securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_INVALID_FORMAT The format of raw_key is not valid
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_remove_alias()
 * @see ckmc_get_key()
 * @see ckmc_get_key_alias_list()
 * @see #ckmc_key_s
 * @see #ckmc_policy_s
 */
int ckmc_save_key(const char *alias, const ckmc_key_s key, const ckmc_policy_s policy);


/**
 * @deprecated Deprecated since @if MOBILE 2.4. @elseif WEARABLE 3.0. @endif [Use ckmc_remove_alias() instead]
 * @brief Removes a key from key manager.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks To remove key, client must have remove permission to the specified key.
 * @remarks The key owner can remove by default.
 * @param[in] alias The name of a key to be removed
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_key()
 * @see ckmc_get_key()
 * @see ckmc_get_key_alias_list()
 */
int ckmc_remove_key(const char *alias)
TIZEN_DEPRECATED_API;


/**
 * @brief Gets a key from key manager.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only data stored by the client.
 * @remarks You must destroy the newly created @a ppkey by calling ckmc_key_free() if it is no longer needed.
 * @param[in] alias The name of a key to retrieve
 * @param[in] password The password used in decrypting a key value \n
 *                     If password of policy is provided in ckmc_save_key(), the same password should be provided
 * @param[out] ppkey The pointer to a newly created ckmc_key_s handle
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Decryption failed because password is incorrect
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_key()
 * @see ckmc_remove_alias()
 * @see ckmc_get_key_alias_list()
 */
int ckmc_get_key(const char *alias, const char *password, ckmc_key_s **ppkey);


/**
 * @brief Gets all the alias of keys that the client can access.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only data stored by the client.
 * @remarks You must destroy the newly created @a ppalias_list by calling ckmc_alias_list_all_free()
 *          if it is no longer needed.
 * @param[out] ppalias_list The pointer to a newly created ckmc_alias_list_s handle containing all
 *                          available alias of keys \n
 *                          If there is no available key alias, *ppalias_list will be null
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_key()
 * @see ckmc_remove_alias()
 * @see ckmc_get_key()
 */
int ckmc_get_key_alias_list(ckmc_alias_list_s **ppalias_list);



/**
 * @brief Stores a certificate inside key manager based on the provided policy.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0
 * @remarks The certificate's binary value will be converted and saved as binary DER encoded certificates.
 * @param[in] alias The name of a certificate to be stored
 * @param[in] cert The certificate's binary value to be stored
 * @param[in] policy The policy about how to store a certificate securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_INVALID_FORMAT The format of raw_cert is not valid
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_remove_alias()
 * @see ckmc_get_cert()
 * @see ckmc_get_cert_alias_list()
 * @see #ckmc_cert_s
 * @see #ckmc_policy_s
 */
int ckmc_save_cert(const char *alias, const ckmc_cert_s cert, const ckmc_policy_s policy);


/**
 * @deprecated Deprecated since @if MOBILE 2.4. @elseif WEARABLE 3.0. @endif [Use ckmc_remove_alias() instead]
 * @brief Removes a certificate from key manager.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks To remove certificate, client must have remove permission to the specified certificate.
 * @remarks The key owner can remove by default.
 * @param[in] alias The name of a certificate to be removed
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_cert()
 * @see ckmc_get_cert()
 * @see ckmc_get_cert_alias_list()
 */
int ckmc_remove_cert(const char *alias)
TIZEN_DEPRECATED_API;


/**
 * @brief Gets a certificate from key manager.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only certificate stored by the client.
 * @remarks A DER encoded certificate will be returned as a return value.
 * @remarks You must destroy the newly created @a ppcert by calling ckmc_cert_free() if it is no longer needed.
 * @param[in] alias The name of a certificate to retrieve
 * @param[in] password The password used in decrypting a certificate value \n
 *                     If password of policy is provided in ckmc_save_cert(), the same password
 *                     should be provided
 * @param[out] ppcert The pointer to a newly created ckmc_cert_s handle
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exists
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Decryption failed because password is incorrect
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_cert()
 * @see ckmc_remove_alias()
 * @see ckmc_get_cert_alias_list()
 */
int ckmc_get_cert(const char *alias, const char *password, ckmc_cert_s **ppcert);


/**
 * @brief Gets all alias of certificates which the client can access.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only data stored by the client.
 * @remarks You must destroy the newly created @a ppalias_list by calling ckmc_alias_list_all_free() if it is no longer needed.
 * @param[out] ppalias_list The pointer to a newly created ckmc_alias_list_s handle containing all available alias of keys \n
 *                          If there is no available key alias, *ppalias_list will be null
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_cert()
 * @see ckmc_remove_alias()
 * @see ckmc_get_cert()
 */
int ckmc_get_cert_alias_list(ckmc_alias_list_s **ppalias_list);


/**
 * @brief Stores PKCS12's contents inside key manager based on the provided policies. All items from the PKCS12 will use the same alias.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @param[in] alias The name of a data to be stored
 * @param[in] pkcs Pointer to the pkcs12 structure to be saved
 * @param[in] key_policy The policy about how to store pkcs's private key
 * @param[in] cert_policy The policy about how to store pkcs's certificate
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_remove_alias()
 * @see ckmc_get_pkcs12()
 * @see ckmc_get_data_alias_list()
 * @see ckmc_pkcs12_load()
 * @see #ckmc_pkcs12_s
 * @see #ckmc_policy_s
 */
int ckmc_save_pkcs12(const char *alias, const ckmc_pkcs12_s *pkcs, const ckmc_policy_s key_policy,const ckmc_policy_s cert_policy);


/**
 * @brief Gets a pkcs12 from key manager.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only data stored by the client.
 * @remarks You must destroy the newly created @a pkcs12 by calling ckmc_pkcs12_free() if it is no longer needed.
 * @param[in] alias The name of a data to retrieve
 * @param[in] key_password Password that was used to encrypt privateKey (may be NULL)
 * @param[in] cert_password Password used to encrypt certificates (may be NULL)
 * @param[out] pkcs12 The pointer to a newly created ckmc_pkcs12_s handle
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED key_password or cert_password does not match with password used to encrypt data
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_pkcs12()
 * @see ckmc_remove_alias()
 */
int ckmc_get_pkcs12(const char *alias, const char *key_password, const char *cert_password, ckmc_pkcs12_s **pkcs12);


/**
 * @brief Stores a data inside key manager based on the provided policy.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @param[in] alias The name of a data to be stored
 * @param[in] data The binary value to be stored
 * @param[in] policy The policy about how to store a data securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_remove_alias()
 * @see ckmc_get_data()
 * @see ckmc_get_data_alias_list()
 * @see #ckmc_raw_buffer_s
 * @see #ckmc_policy_s
 */
int ckmc_save_data(const char *alias, ckmc_raw_buffer_s data, const ckmc_policy_s policy);


/**
 * @deprecated Deprecated since @if MOBILE 2.4. @elseif WEARABLE 3.0. @endif [Use ckmc_remove_alias() instead]
 * @brief Removes a data from key manager.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks To remove data, client must have remove permission to the specified data object.
 * @remarks The data owner can remove by default.
 * @param[in] alias The name of a data to be removed
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_data()
 * @see ckmc_get_data()
 * @see ckmc_get_data_alias_list()
 */
int ckmc_remove_data(const char *alias)
TIZEN_DEPRECATED_API;


/**
 * @brief Gets a data from key manager.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only data stored by the client.
 * @remarks You must destroy the newly created @a ppdata by calling ckmc_buffer_free() if it is no longer needed.
 * @param[in] alias The name of a data to retrieve
 * @param[in] password The password used in decrypting a data value \n
 *                     If password of policy is provided in ckmc_save_data(), the same password
 *                     should be provided
 * @param[out] ppdata The pointer to a newly created ckmc_raw_buffer_s handle
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Decryption failed because password is incorrect
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_data()
 * @see ckmc_remove_alias()
 * @see ckmc_get_data_alias_list()
 */
int ckmc_get_data(const char *alias, const char *password, ckmc_raw_buffer_s **ppdata);


/**
 * @brief Gets all alias of data which the client can access.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks A client can access only data stored by the client.
 * @remarks You must destroy the newly created @a ppalias_list by calling ckmc_alias_list_all_free() if it is no longer needed.
 * @param[out] ppalias_list The pointer to a newly created ckmc_alias_list_s handle containing all available alias of keys \n
 *                          If there is no available key alias, *ppalias_list will be null
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_data()
 * @see ckmc_remove_alias()
 * @see ckmc_get_data()
 */
int ckmc_get_data_alias_list(ckmc_alias_list_s **ppalias_list);


/**
 * @brief Creates RSA private/public key pair and stores them inside key manager based on each policy.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks If password in the policy is provided, the key is additionally encrypted with the password in the policy.
 * @param[in] size The size of key strength to be created \n
 *                 @c 1024, @c 2048, and @c 4096 are supported
 * @param[in] private_key_alias The name of private key to be stored
 * @param[in] public_key_alias The name of public key to be stored
 * @param[in] policy_private_key The policy about how to store a private key securely
 * @param[in] policy_public_key The policy about how to store a public key securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_DB_ERROR Failed due to other DB transaction unexpectedly
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_create_key_pair_dsa()
 * @see ckmc_create_key_pair_ecdsa()
 * @see ckmc_create_signature()
 * @see ckmc_verify_signature()
 */
int ckmc_create_key_pair_rsa(const size_t size, const char *private_key_alias, const char *public_key_alias, const ckmc_policy_s policy_private_key, const ckmc_policy_s policy_public_key);


/**
 * @brief Creates DSA private/public key pair and stores them inside key manager based on each policy.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks If password in the policy is provided, the key is additionally encrypted with the password in the policy.
 * @param[in] size The size of key strength to be created \n
 *                 @c 1024, @c 2048, @c 3072 and @c 4096 are supported
 * @param[in] private_key_alias The name of private key to be stored
 * @param[in] public_key_alias The name of public key to be stored
 * @param[in] policy_private_key The policy about how to store a private key securely
 * @param[in] policy_public_key The policy about how to store a public key securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_DB_ERROR Failed due to other DB transaction unexpectedly
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_create_key_pair_rsa()
 * @see ckmc_create_key_pair_ecdsa()
 * @see ckmc_create_signature()
 * @see ckmc_verify_signature()
 */
int ckmc_create_key_pair_dsa(const size_t size, const char *private_key_alias, const char *public_key_alias, const ckmc_policy_s policy_private_key, const ckmc_policy_s policy_public_key);


/**
 * @brief Creates ECDSA private/public key pair and stores them inside key manager based on each policy.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks If password in the policy is provided, the key is additionally encrypted with the password in the policy.
 * @param[in] type The type of elliptic curve of ECDSA
 * @param[in] private_key_alias The name of private key to be stored
 * @param[in] public_key_alias The name of public key to be stored
 * @param[in] policy_private_key The policy about how to store a private key securely
 * @param[in] policy_public_key The policy about how to store a public key securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_DB_ERROR Failed due to other DB transaction unexpectedly
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_create_key_pair_rsa()
 * @see ckmc_create_key_pair_dsa()
 * @see ckmc_create_signature()
 * @see ckmc_verify_signature()
 * @see #ckmc_ec_type_e
 */
int ckmc_create_key_pair_ecdsa(const ckmc_ec_type_e type, const char *private_key_alias, const char *public_key_alias, const ckmc_policy_s policy_private_key, const ckmc_policy_s policy_public_key);


/**
 * @brief Creates AES key and stores it inside key manager based on the policy.
 * @since_tizen 3.0
 * @remarks If password in the policy is provided, the key is additionally encrypted with the password in the policy.
 * @param[in] size The size of key strength to be created \n
 *                 @c 128, @c 192 and @c 256 are supported
 * @param[in] key_alias The name of key to be stored
 * @param[in] key_policy The policy about how to store the key securely
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ALIAS_EXISTS Alias already exists
 * @retval #CKMC_ERROR_DB_ERROR Failed due to other DB transaction unexpectedly
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_create_key_pair_rsa()
 * @see ckmc_create_key_pair_dsa()
 * @see ckmc_create_key_pair_ecdsa()
 * @see #ckmc_policy_s
 */
int ckmc_create_key_aes(size_t size, const char *key_alias, ckmc_policy_s key_policy);


/**
 * @brief Creates a signature on a given message using a private key and returns the signature.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks If password of policy is provided during storing a key, the same password should be provided.
 * @remarks You must destroy the newly created @a ppsignature by calling ckmc_buffer_free() if it is no longer needed.
 * @param[in] private_key_alias The name of private key
 * @param[in] password The password used in decrypting a private key value
 * @param[in] message The message that is signed with a private key
 * @param[in] hash The hash algorithm used in creating signature
 * @param[in] padding The RSA padding algorithm used in creating signature \n
 *                    It is used only when the signature algorithm is RSA. If
 *                    @a padding is CKMC_NONE_PADDING you must use CKMC_HASH_NONE
 *                    and the message must be equal to key length
 * @param[out] ppsignature The pointer to a newly created signature \n
 *                         If an error occurs, @a *ppsignature will be null
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Decryption failed because password is incorrect
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_create_key_pair_rsa()
 * @see ckmc_create_key_pair_ecdsa()
 * @see ckmc_verify_signature()
 * @see ckmc_buffer_free()
 * @see #ckmc_hash_algo_e
 * @see #ckmc_rsa_padding_algo_e
 */
int ckmc_create_signature(const char *private_key_alias, const char *password, const ckmc_raw_buffer_s message, const ckmc_hash_algo_e hash, const ckmc_rsa_padding_algo_e padding, ckmc_raw_buffer_s **ppsignature);


/**
 * @brief Verifies a given signature on a given message using a public key and returns the signature status.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks If password of policy is provided during storing a key, the same password should be provided.
 * @param[in] public_key_alias The name of public key
 * @param[in] password The password used in decrypting a public key value
 * @param[in] message The input on which the signature is created
 * @param[in] signature The signature that is verified with public key
 * @param[in] hash The hash algorithm used in verifying signature
 * @param[in] padding The RSA padding algorithm used in verifying signature \n
 *                    It is used only when the signature algorithm is RSA. If
 *                    @a padding is CKMC_NONE_PADDING you must use CKMC_HASH_NONE
 *                    and the message must be equal to key length
 * @return @c 0 on success and the signature is valid,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_VERIFICATION_FAILED The signature is invalid
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Decryption failed because password is incorrect
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_create_key_pair_rsa()
 * @see ckmc_create_key_pair_ecdsa()
 * @see ckmc_verify_signature()
 * @see #ckmc_hash_algo_e
 * @see #ckmc_rsa_padding_algo_e
 */
int ckmc_verify_signature(const char *public_key_alias, const char *password, const ckmc_raw_buffer_s message, const ckmc_raw_buffer_s signature, const ckmc_hash_algo_e hash, const ckmc_rsa_padding_algo_e padding);


/**
 * @brief Verifies a certificate chain and returns that chain.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks The trusted root certificate of the chain should exist in the system's certificate storage.
 * @remarks You must destroy the newly created @a ppcert_chain_list by calling ckmc_cert_list_all_free() if it is no longer needed.
 * @param[in] cert The certificate to be verified
 * @param[in] untrustedcerts The untrusted CA certificates to be used in verifying a certificate chain
 * @param[out] ppcert_chain_list The pointer to a newly created certificate chain's handle \n
 *                               If an error occurs, @a *ppcert_chain_list will be null
 * @return @c 0 on success and the signature is valid,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_VERIFICATION_FAILED The certificate chain is not valid
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_INVALID_FORMAT The format of certificate is not valid
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Decryption failed because password is incorrect
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_cert_list_all_free()
 */
int ckmc_get_cert_chain(const ckmc_cert_s *cert, const ckmc_cert_list_s *untrustedcerts, ckmc_cert_list_s **ppcert_chain_list);


/**
 * @deprecated Deprecated since @if MOBILE 2.4. @elseif WEARABLE 3.0. @endif [Use ckmc_get_cert_chain() instead]
 * @brief Verifies a certificate chain using an alias list of untrusted certificates and return that chain.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks The trusted root certificate of the chain should exist in the system's certificate storage.
 * @remarks You must destroy the newly created @a ppcert_chain_list by calling ckmc_cert_list_all_free() if it is no longer needed.
 * @remarks @a untrustedcerts shouldn't be protected with optional password.
 * @param[in] cert The certificate to be verified
 * @param[in] untrustedcerts The alias list of untrusted CA certificates stored in key manager to be used in verifying a certificate chain
 * @param[out] ppcert_chain_list The pointer to a newly created certificate chain's handle \n
 *                               If an error occurs, @a *ppcert_chain_list will be null
 * @return @c 0 on success and the signature is valid,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_VERIFICATION_FAILED The certificate chain is not valid
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_INVALID_FORMAT The format of certificate is not valid
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Some certificates were encrypted with password and could not be used
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_get_cert_chain()
 * @see ckmc_cert_list_all_free()
 */
int ckmc_get_cert_chain_with_alias(const ckmc_cert_s *cert, const ckmc_alias_list_s *untrustedcerts, ckmc_cert_list_s **ppcert_chain_list) TIZEN_DEPRECATED_API;


/**
 * @brief Verifies a certificate chain and returns that chain using user-entered, trusted, and untrusted CA certificates.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks If the trusted root certificates are provided as a user input, these certificates do not need to exist in the system's certificate storage.
 * @remarks You must destroy the newly created @a ppcert_chain_list by calling ckmc_cert_list_all_free() if it is no longer needed.
 * @param[in] cert The certificate to be verified
 * @param[in] untrustedcerts The untrusted CA certificates to be used in verifying a certificate chain
 * @param[in] trustedcerts The trusted CA certificates to be used in verifying a certificate chain
 * @param[in] use_trustedsystemcerts The flag indicating the use of the trusted root certificates in the system's certificate storage
 * @param[out] ppcert_chain_list The pointer to a newly created certificate chain's handle \n
 *                               If an error occurs, @a *ppcert_chain_list will be null
 * @return @c 0 on success and the signature is valid,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_VERIFICATION_FAILED The certificate chain is not valid
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_INVALID_FORMAT The format of certificate is not valid
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_cert_list_all_free()
 */
int ckmc_get_cert_chain_with_trustedcert(const ckmc_cert_s *cert, const ckmc_cert_list_s *untrustedcerts, const ckmc_cert_list_s *trustedcerts, const bool use_trustedsystemcerts, ckmc_cert_list_s **ppcert_chain_list);


/**
 * @brief Perform OCSP that checks certificate is whether revoked or not.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @privlevel public
 * @privilege %http://tizen.org/privilege/internet
 * @remarks %http://tizen.org/privilege/internet (public level privilege) is required to use this API instead of %http://tizen.org/privilege/keymanager (public level privilege) since 3.0.
 * @param[in] pcert_chain_list Valid certificate chain to perform OCSP check
 * @param[out] ocsp_status The pointer to status result of OCSP check
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_NOT_SUPPORTED Device needed to run API is not supported
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @pre @a pcert_chain_list is created with ckmc_get_certificate_chain() or
 *      ckmc_get_certificate_chain_with_alias().
 * @see ckmc_get_cert_chain())
 * @see ckmc_cert_list_all_free()
 */
int ckmc_ocsp_check(const ckmc_cert_list_s *pcert_chain_list, ckmc_ocsp_status_e *ocsp_status);


/**
 * @deprecated Deprecated since @if MOBILE 2.4. @elseif WEARABLE 3.0. @endif [Use ckmc_set_permission() instead]
 * @brief Allows another application to access client's application data.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks Data identified by @a alias should exist.
 * @param[in] alias Data alias for which access will be granted
 * @param[in] accessor Package id of the application that will gain access rights
 * @param[in] granted Rights granted for @a accessor application
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_deny_access()
 */
int ckmc_allow_access(const char *alias, const char *accessor, ckmc_access_right_e granted) TIZEN_DEPRECATED_API;


/**
 * @brief Allows another application to access client's application data.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks Data identified by @a alias should exist.
 * @param[in] alias Data alias for which access will be granted
 * @param[in] accessor Package id of the application that will gain access rights
 * @param[in] permissions Mask of permissions granted for @a accessor application
 *                        (#ckmc_permission_e)
 *                        (previous permission mask will be replaced with the new mask value)
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 */
int ckmc_set_permission(const char *alias, const char *accessor, int permissions);


/**
 * @deprecated Deprecated since @if MOBILE 2.4. @elseif WEARABLE 3.0. @endif [Use ckmc_set_permission() instead]
 * @brief Revokes another application's access to client's application data.
 * @since_tizen 2.3
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks Data identified by @a alias should exist.
 * @remarks Only access previously granted with ckmc_allow_access can be revoked.
 * @param[in] alias Data alias for which access will be revoked
 * @param[in] accessor Package id of the application that will lose access rights
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid or the @a accessor doesn't have access to @a alias
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_allow_access()
 * @see ckmc_set_permission()
 */
int ckmc_deny_access(const char *alias, const char *accessor) TIZEN_DEPRECATED_API;


/**
 * @brief Removes an entry (no matter of type) from the key manager.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks %http://tizen.org/privilege/keymanager (public level privilege) is no longer required to use this API since 3.0.
 * @remarks To remove item, client must have remove permission to the specified item.
 * @remarks The item owner can remove by default.
 * @param[in] alias Item alias to be removed
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to a database error
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_save_key()
 * @see ckmc_save_cert()
 * @see ckmc_save_data()
 * @see ckmc_save_pkcs12()
 * @see ckmc_create_key_pair_rsa()
 * @see ckmc_create_key_pair_dsa()
 * @see ckmc_create_key_pair_ecdsa()
 */
int ckmc_remove_alias(const char *alias);


/**
 * @brief Encrypts data using selected key and algorithm.
 * @since_tizen 3.0
 * @remarks Key identified by @a key_alias should exist.
 * @param[in] params Algorithm parameter list handle. See #ckmc_param_list_h and
 *                   #ckmc_algo_type_e for details
 * @param[in] key_alias Alias of the key to be used for encryption
 * @param[in] password The password used in decrypting a key value \n
 *                     If password of the policy is provided in ckmc_save_key(), the same
 *                     password should be provided
 * @param[in] decrypted Data to be encrypted. In case of AES algorithm there are no restrictions on the size of data,
 *                      if S/W backend is used. If module uses TEE backend (since Tizen 5.0 on chosen images),
 *                      maximum size of data is implementation-specific and at least 500 kB.
 *                      For RSA the size must be smaller or equal to key size in bytes - 42.
 *                      Example: for 1024 RSA key the maximum data size is 1024/8 - 42 = 86.
 * @param[out] ppencrypted Encrypted data (some algorithms may return additional information embedded in encrypted data.
 *                         AES GCM is an example) \n
 *                         The caller is responsible for freeing @a encrypted with ckmc_buffer_free()
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid (missing or invalid
 *                                       mandatory algorithm parameter or RSA data too long, decrypted = NULL,
 *                                       ppencrypted = NULL)
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Key with given alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Key decryption failed because password is incorrect
 * @retval #CKMC_ERROR_SERVER_ERROR Too big data size or unsupported GCM mode (32 and 64 bit tag lengths not supported on TEE backend) or internal error
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_buffer_free()
 * @see ckmc_param_list_new()
 * @see ckmc_param_list_free()
 * @see ckmc_param_list_set_integer()
 * @see ckmc_param_list_set_buffer()
 * @see ckmc_generate_new_params()
 * @see #ckmc_param_list_h
 * @see #ckmc_param_name_e
 * @see #ckmc_algo_type_e
 */
int ckmc_encrypt_data(ckmc_param_list_h params, const char *key_alias, const char *password, const ckmc_raw_buffer_s decrypted, ckmc_raw_buffer_s **ppencrypted);


/**
 * @brief Decrypts data using selected key and algorithm.
 * @since_tizen 3.0
 * @remarks Key identified by @a key_alias should exist.
 * @param[in] params Algorithm parameter list handle. You should use the same parameters that were used for encryption.
 *                   See #ckmc_param_list_h and #ckmc_algo_type_e for details
 * @param[in] key_alias Alias of the key to be used for encryption
 * @param[in] password The password used in decrypting a key value \n
 *                     If password of the policy is provided in ckmc_save_key(), the same password should be provided
 * @param[in] encrypted Data to be decrypted (some algorithms may require additional information embedded in encrypted data. AES GCM is an example)
 *                      Since Tizen 5.0, on chosen images where module is using TEE backend, data size is limited to at least 500 kB
 *                      (TEE implementation-specific).
 * @param[out] ppdecrypted Decrypted data \n
 *                         The caller is responsible for freeing @a decrypted with ckmc_buffer_free()
 * @return @c 0 on success,
 *         otherwise a negative error value
 * @retval #CKMC_ERROR_NONE Successful
 * @retval #CKMC_ERROR_INVALID_PARAMETER Input parameter is invalid (missing or invalid
 *                                       mandatory algorithm parameter, GCM tag authentication failed, key or data is wrong,
 *                                       in case of RSA key is wrong or data too long, encrypted = NULL, ppdecrypted = NULL)
 * @retval #CKMC_ERROR_DB_LOCKED A user key is not loaded in memory (a user is not logged in)
 * @retval #CKMC_ERROR_DB_ERROR Failed due to the error with unknown reason
 * @retval #CKMC_ERROR_DB_ALIAS_UNKNOWN Key with given alias does not exist
 * @retval #CKMC_ERROR_PERMISSION_DENIED Failed to access key manager
 * @retval #CKMC_ERROR_AUTHENTICATION_FAILED Key decryption failed because password is incorrect
 * @retval #CKMC_ERROR_SERVER_ERROR Too big data size or unsupported GCM mode (32 and 64 bit tag lengths not supported on TEE backend) or internal error
 * @pre User is already logged in and the user key is already loaded into memory in plain text form.
 * @see ckmc_buffer_free()
 * @see ckmc_param_list_new()
 * @see ckmc_param_list_free()
 * @see ckmc_param_list_set_integer()
 * @see ckmc_param_list_set_buffer()
 * @see ckmc_generate_new_params()
 * @see #ckmc_param_list_h
 * @see #ckmc_param_name_e
 * @see #ckmc_algo_type_e
 */
int ckmc_decrypt_data(ckmc_param_list_h params, const char *key_alias, const char *password, const ckmc_raw_buffer_s encrypted, ckmc_raw_buffer_s **ppdecrypted);


#ifdef __cplusplus
}
#endif


/**
 * @}
 */


#endif /* __TIZEN_CORE_CKMC_MANAGER_H */
