/**
 * Copyright (c) 2011-2018 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#ifndef __TIZEN_UIFW_MULTI_ASSISTANT_H__
#define __TIZEN_UIFW_MULTI_ASSISTANT_H__

#include <multi_assistant_common.h>


/**
 * @defgroup CAPI_UIX_MULTI_ASSISTANT_CLIENT_MODULE Multi assistant client
 * @ingroup CAPI_UIX_MULTI_ASSISTANT_MODULE
 * @brief The @ref CAPI_UIX_MULTI_ASSISTANT_CLIENT_MODULE API provides functions for getting queries from multi assistant service and sending responses.
 * @{
 */


#ifdef __cplusplus
extern "C"
{
#endif



/**
 * @brief Initializes multi-assistant client.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_OPERATION_FAILED Operation failed
 * @retval #MA_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @post If this function is called, the state will be #MA_STATE_INITIALIZED.
 * @see ma_deinitialize()
 */
int ma_initialize(void);

/**
 * @brief Deinitializes multi-assistant client.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @post If this function is called, the state will be #MA_STATE_NONE.
 * @see ma_initialize()
 */
int ma_deinitialize(void);

/**
 * @brief Prepares multi-assistant client.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @post If this function is called, the state will be #MA_STATE_READY.
 * @see ma_unprepare()
 */
int ma_prepare(void);

/**
 * @brief Unprepares multi-assistant client.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @pre The state should be #MA_STATE_READY.
 * @post If this function is called, the state will be #MA_STATE_INITIALIZED.
 * @see ma_prepare()
 */
int ma_unprepare(void);

/**
 * @brief Gets the current state of the multi-assistant client.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @param[out] state The current state
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 */
int ma_get_state(ma_state_e* state);

/**
 * @brief Gets the current language of multi-assistant client.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @remarks The @a language should be released using free().
 * @param[out] language The current language
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 */
int ma_get_current_language(char** language);

/**
 * @brief Gets the recording audio format.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @param[out] rate The audio sampling rate
 * @param[out] channel The audio channel
 * @param[out] audio_type The audio type
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MA_ERROR_OPERATION_FAILED Operation failed
 *
 * @pre The state should be #MA_STATE_READY.
 */
int ma_get_recording_audio_format(int* rate, ma_audio_channel_e* channel, ma_audio_type_e* audio_type);

/**
 * @brief Sets a state changed callback.
 * @since_tizen 5.0
 *
 * @param[in] callback The callback
 * @param[in] user_data The user data passed to the callback function
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_state_changed_cb()
 * @see ma_unset_state_changed_cb()
 */
int ma_set_state_changed_cb(ma_state_changed_cb callback, void* user_data);

/**
 * @brief Unsets a state changed callback.
 * @since_tizen 5.0
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_state_changed_cb()
 * @see ma_set_state_changed_cb()
 */
int ma_unset_state_changed_cb(void);

/**
 * @brief Sets an error callback.
 * @since_tizen 5.0
 *
 * @param[in] callback The callback
 * @param[in] user_data The user data passed to the callback function
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_error_cb()
 * @see ma_unset_errord_cb()
 */
int ma_set_error_cb(ma_error_cb callback, void* user_data);

/**
 * @brief Unsets an error callback.
 * @since_tizen 5.0
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_error_cb()
 * @see ma_set_error_cb()
 */
int ma_unset_error_cb(void);

/**
 * @brief Sets a language changed callback.
 * @since_tizen 5.0
 *
 * @param[in] callback The callback
 * @param[in] user_data The user data passed to the callback function
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_language_changed_cb()
 * @see ma_unset_language_changed_cb()
 */
int ma_set_language_changed_cb(ma_language_changed_cb callback, void* user_data);

/**
 * @brief Unsets a language changed callback.
 * @since_tizen 5.0
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_language_changed_cb()
 * @see ma_set_language_changed_cb()
 */
int ma_unset_language_changed_cb(void);

/**
 * @brief Sets an audio streaming callback.
 * @since_tizen 5.0
 *
 * @param[in] callback The callback
 * @param[in] user_data The user data passed to the callback function
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_audio_streaming_cb()
 * @see ma_unset_audio_streaming_cb()
 */
int ma_set_audio_streaming_cb(ma_audio_streaming_cb callback, void* user_data);

/**
 * @brief Unsets an audio streaming callback.
 * @since_tizen 5.0
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_INVALID_STATE Invalid state
 *
 * @pre The state should be #MA_STATE_INITIALIZED.
 * @see ma_audio_streaming_cb()
 * @see ma_set_audio_streaming_cb()
 */
int ma_unset_audio_streaming_cb(void);

/**
 * @brief Sends ASR(auto speech recognition) results to the multi-assistant service.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @param[in] event The ASR result event (e.g. #MA_ASR_RESULT_EVENT_FINAL_RESULT)
 * @param[in] asr_result The ASR result text
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 */
int ma_send_asr_result(ma_asr_result_event_e event, const char* asr_result);

/**
 * @brief Sends results to the multi-assistant service.
 * @since_tizen 5.0
 * @privlevel public
 * @privilege %http://tizen.org/privilege/recorder
 *
 * @param[in] display_text The text shown on the display
 * @param[in] utterance_text The utterance text
 * @param[in] result_json The result data (JSON format)
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MA_ERROR_NONE Successful
 * @retval #MA_ERROR_NOT_SUPPORTED Not supported
 * @retval #MA_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MA_ERROR_INVALID_PARAMETER Invalid parameter
 */
int ma_send_result(const char* display_text, const char* utterance_text, const char* result_json);


#ifdef __cplusplus
}
#endif

/**
 * @}
 */

#endif /* __TIZEN_UIFW_MULTI_ASSISTANT_H__ */
