/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "simple_tcp_server.h"

#include <stdio.h>
#include <string>
#include <sstream>

#include "ppapi/c/pp_errors.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "logger.h"
#include "net_utils.h"

void SimpleTCPServer::Reset() {
  listening_socket_ = pp::TCPSocket(instance_handle_);
  incoming_socket_ = pp::TCPSocket(instance_handle_);
}

bool SimpleTCPServer::InterfacesAreAvailable() {
  // Check if TCPSocket interface is available.
  if (!pp::TCPSocket::IsAvailable()) {
    Logger::Error("TCPServer: TCPSocket not available");
    return false;
  }

  return true;
}

void SimpleTCPServer::Listen(uint16_t port,
    pp::CompletionCallbackWithOutput<std::string> on_server_accept_callback,
    pp::CompletionCallbackWithOutput<std::string> on_server_receive_callback) {
  listening_socket_ = pp::TCPSocket(instance_handle_);
  if (listening_socket_.is_null()) {
    Logger::Error("TCPServer: Error creating TCPSocket.");
    return;
  }

  Logger::Log("TCPServer: Starting server on port: %d", port);

  // Save callbacks, so they can be used later.
  on_server_accept_callback_ = on_server_accept_callback;
  on_server_receive_callback_ = on_server_receive_callback;

  // Attempt to listen on all interfaces (0.0.0.0)
  // on the given port number.
  PP_NetAddress_IPv4 ipv4_addr = { Htons(port), { 0 } };
  pp::NetAddress addr(instance_handle_, ipv4_addr);
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPServer::OnBindCompletion);

  // We are binding to the specific address (this operation is
  // optional).
  int32_t rtn = listening_socket_.Bind(addr, callback);
  if (rtn != PP_OK_COMPLETIONPENDING) {
    Logger::Error("TCPServer: Error binding listening socket.");
    return;
  }
}

void SimpleTCPServer::Write(const std::string& message,
    pp::CompletionCallbackWithOutput<std::string> on_server_receive_callback) {

  // Save the receive callback for later use.
  on_server_receive_callback_ = on_server_receive_callback;
  pp::CompletionCallback callback =
        callback_factory_.NewCallback(&SimpleTCPServer::OnWriteCompletion);

  // Write a message on to the socket.
  int32_t result = incoming_socket_.Write(message.c_str(), message.length(),
                                          callback);
  if (result != PP_OK_COMPLETIONPENDING)
    Logger::Error("TCPServer: Write failed: %d", result);
}

void SimpleTCPServer::TryRead() {
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPServer::OnReadCompletion);
  // Invoke read on socket. The receive_buffer_ will be filled
  // with the message content when the callback is called
  // without any errors.
  incoming_socket_.Read(receive_buffer_, kBufferSize, callback);
}

void SimpleTCPServer::TryAccept() {
  pp::CompletionCallbackWithOutput<pp::TCPSocket> callback =
      callback_factory_.NewCallbackWithOutput(
          &SimpleTCPServer::OnAcceptCompletion);
  // Invoke Accept. Callback will be called when we receive
  // incoming connection.
  listening_socket_.Accept(callback);
}

void SimpleTCPServer::OnBindCompletion(int32_t result) {
  if (result != PP_OK) {
    Logger::Error("TCPServer: Bind failed with: %d", result);
    return;
  }

  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPServer::OnListenCompletion);
  // After successful Bind, we start automatically listening.
  int32_t rtn = listening_socket_.Listen(kBacklog, callback);
  if (rtn != PP_OK_COMPLETIONPENDING) {
    Logger::Error("TCPServer: Error listening on server socket.");
    return;
  }
}

void SimpleTCPServer::OnListenCompletion(int32_t result) {
  if (result != PP_OK) {
    Logger::Error("TCPServer: Listen failed with: %d", result);
    return;
  }

  // We are getting precise address on which we are listening.
  pp::NetAddress addr = listening_socket_.GetLocalAddress();
  Logger::Log("TCPServer: Listening on: %s",
            addr.DescribeAsString(true).AsString().c_str());

  // We try to accept all incoming connections.
  TryAccept();
}

void SimpleTCPServer::OnAcceptCompletion(int32_t result, pp::TCPSocket socket) {
  if (result != PP_OK) {
    if(result == PP_ERROR_ABORTED) {
      // We consume aborted callback. This can occur when current server
      // is being shut down.
      Logger::Log("TCPServer: Consuming old Accept callback");
      return;
    }
    // We run the callback together with the failed result.
    on_server_accept_callback_.Run(result);
    return;
  }

  // We are getting endpoint address for newly accepted connection.
  pp::NetAddress addr = socket.GetRemoteAddress();
  // We fill out message for parameter, which will be passed to
  // callback funtion.
  *on_server_accept_callback_.output() = addr.DescribeAsString(true).AsString();
  // We run the callback together with the result.
  on_server_accept_callback_.Run(PP_OK);

  // We save the newly connected client.
  incoming_socket_ = socket;

  // Automatically try to read an incoming message.
  TryRead();
}

void SimpleTCPServer::OnReadCompletion(int32_t result) {
  if (result <= 0) {
    if(result == PP_ERROR_ABORTED) {
      // We consume aborted callback. This can occur when current server
      // is being shut down.
      Logger::Log("TCPServer: Consuming old Read callback");
      return;
    }
    if (result == 0)
      Logger::Error("TCPServer: client disconnected");
    else
      Logger::Error("TCPServer: Read failed: %d", result);
  } else {
    Logger::Log("TCPServer: Read %d bytes", result);

    // We set our message as a parameter, which will be
    // passed to the callback function.
    *on_server_receive_callback_.output() = std::string(receive_buffer_, result);
  }

  // We run the callback together with the result.
  on_server_receive_callback_.Run(result);
}

void SimpleTCPServer::OnWriteCompletion(int32_t result) {
  if (result < 0) {
    Logger::Error("TCPServer: Write failed: %d", result);
    return;
  }

  Logger::Log("TCPServer: Wrote %d bytes", result);

  // Try and read more bytes from the socket.
  TryRead();
}
