/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "simple_udp_client.h"

#include <stdio.h>
#include <string>
#include <cstring>
#include <sstream>

#include "ppapi/c/pp_errors.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "logger.h"
#include "net_utils.h"

bool SimpleUDPClient::InterfacesAreAvailable() {
  // Check if UDPSocket interface is available.
  if (!pp::UDPSocket::IsAvailable()) {
    Logger::Error("UDPSocket not available");
    return false;
  }

  // Check if HostResolver interface is available.
  if (!pp::HostResolver::IsAvailable()) {
    Logger::Error("HostResolver not available");
    return false;
  }

  return true;
}

void SimpleUDPClient::Bind(const std::string& destination_host,
    pp::CompletionCallbackWithOutput<std::string> receive_from_callback) {

  // Check if this socket is already bound.
  if (IsBound()) {
    Logger::Error("UDPClient: Already bound.");
    return;
  }

  // Create new UDPSocket which will be used for
  // future connections.
  udp_socket_ = pp::UDPSocket(instance_handle_);
  if (udp_socket_.is_null()) {
    Logger::Error("UDPClient: Error creating UDPSocket.");
    return;
  }

  // Create new HostResolver to resolve IP from
  // a given host name.
  resolver_ = pp::HostResolver(instance_handle_);
  if (resolver_.is_null()) {
    Logger::Error("UDPClient: Error creating HostResolver.");
    return;
  }

  int port;
  std::string hostname;
  ParseHost(destination_host, kPort, port, hostname);

  // We remember on_receive_from_callback callback so
  // it can be called up after receiving message.
  on_receive_from_callback_ = receive_from_callback;
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleUDPClient::OnResolveCompletion);
  PP_HostResolver_Hint hint = { PP_NETADDRESS_FAMILY_UNSPECIFIED, 0 };
  // We resolve hostname to IP format.
  resolver_.Resolve(hostname.c_str(), port, hint, callback);
  Logger::Log("UDPClient: Resolving ...");
}

void SimpleUDPClient::Close() {
  // We can close only bound socket.
  if (!IsBound()) {
    Logger::Error("UDPClient: Not bound.");
    return;
  }

  // Close opened socket.
  udp_socket_.Close();
  udp_socket_ = pp::UDPSocket();
  Logger::Log("UDPClient: Closed connection.");
}

void SimpleUDPClient::SendTo(const std::string& message) {
  // First we check if we are bound.
  if (!IsBound()) {
    Logger::Error("UDPClient: Not bound.");
    return;
  }

  // We are checking if we are not already sending
  // a message.
  if (send_outstanding_) {
    Logger::Error("UDPClient: Already sending.");
    return;
  }

  uint32_t size = message.size();
  const char* data = message.c_str();
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleUDPClient::OnSendToCompletion);
  int32_t result;

  // We send to remote host.
  result = udp_socket_.SendTo(data, size, remote_host_, callback);
  if (result < 0) {
    if (result == PP_OK_COMPLETIONPENDING) {
      Logger::Log("UDPClient: Sending bytes: %d", size);
      send_outstanding_ = true;
    } else {
      Logger::Error("UDPClient: Send returned error: %d", result);
    }
  } else {
    Logger::Log("UDPClient: Sent bytes synchronously: %d", result);
  }
}

bool SimpleUDPClient::IsBound() {
  if (!udp_socket_.is_null())
    return true;
  return false;
}


void SimpleUDPClient::Receive(
    pp::CompletionCallbackWithOutput<std::string> on_receive_from_callback) {
  // We save on_receive_callback callback so we
  // can call it when we receive message.
  on_receive_from_callback_ = on_receive_from_callback;
  memset(receive_buffer_, 0, kBufferSize);
  pp::CompletionCallbackWithOutput<pp::NetAddress> callback =
      callback_factory_.NewCallbackWithOutput(
          &SimpleUDPClient::OnReceiveFromCompletion);
  // We invoke RecvFrom method on socket.
  // The receive_buffer_ variable will be filled with
  // received data after successful callback completion.
  udp_socket_.RecvFrom(receive_buffer_, kBufferSize, callback);
}

// This callback is called when resolving hostname
// have been finished.
void SimpleUDPClient::OnResolveCompletion(int32_t result) {
  if (result != PP_OK) {
    Logger::Error("UDPClient: Resolve failed.");
    return;
  }

  // We are getting resolved IP address from provided
  // host name in Bind method.
  pp::NetAddress addr = resolver_.GetNetAddress(0);
  Logger::Log(std::string("UDPClient: Resolved: ") +
              addr.DescribeAsString(true).AsString());

  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleUDPClient::OnBindCompletion);

  Logger::Log("UDPClient: Binding ...");
  // We save remote_host_ address for future use in SendTo
  // method.
  remote_host_ = addr;
  PP_NetAddress_IPv4 ipv4_addr = { 0, { 0 } };
  // Before we start receiving, we need to bind
  // socket to specific address and port.
  udp_socket_.Bind(pp::NetAddress(instance_handle_, ipv4_addr), callback);
}

// This callback is called upon Bind completion.
void SimpleUDPClient::OnBindCompletion(int32_t result) {
  if (result != PP_OK) {
    Logger::Error("UDPClient: Connection failed: %d", result);
    return;
  }

  // We are getting precise address to which we are bound.
  pp::NetAddress addr = udp_socket_.GetBoundAddress();
    Logger::Log(std::string("UDPClient: Bound to: ") +
                addr.DescribeAsString(true).AsString());
  // Invoke receive message.
  Receive(on_receive_from_callback_);
}

// This callback is called upon receiving message on socket
void SimpleUDPClient::OnReceiveFromCompletion(int32_t result,
                                              pp::NetAddress source) {
  UNUSED(source);

  if (result > 0) {
    // The receive_buffer_ variable contains our received message.
    std::string message(receive_buffer_, result);
    // We fill out message for parameter, which will be passed to
    // callback funtion.
    *on_receive_from_callback_.output() = message;
  }

  // We run callback together with result.
  on_receive_from_callback_.Run(result);
}

// This callback is called upon completion of sending message
void SimpleUDPClient::OnSendToCompletion(int32_t result) {
  if (result < 0) {
    Logger::Error("UDPClient: Send failed with: %d", result);
  } else {
    Logger::Log("UDPClient: Sent bytes: %d", result);
  }
  send_outstanding_ = false;
}
