/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "threads_and_callbacks.h"

#include <random>
#include <math.h>
#include <string>
#include <sstream>
#include <functional>

#include <ppapi/cpp/var.h>
#include <ppapi/cpp/instance_handle.h>

using namespace pp;

namespace {

const std::string kCalculatePiCommand  = "calculate_PI:";
const std::string KDownloadUrlCommand = "download_URL:";
const std::string KPiPrefix = "PI calculated by MonteCarlo method with ";

} // namespace

ThreadsAndCallbacksInstance::ThreadsAndCallbacksInstance(PP_Instance instance)
    : Instance(instance)
    , simple_thread_pi_(InstanceHandle(instance))
    , factory_(this) {
}

bool ThreadsAndCallbacksInstance::Init(
    uint32_t argc, const char* argn[], const char* argv[]) {

  // pp::SimpleThread needs to be started. It creates a thread with
  // a message loop and runs this loop.
  return simple_thread_pi_.Start();
}

// A HandleMessage function is always called in a main PPAPI thread.
void ThreadsAndCallbacksInstance::HandleMessage(const Var& message) {
  if (!message.is_string())
    return;
  std::string text = message.AsString();
  if (text.find(kCalculatePiCommand) != std::string::npos) {
    int iterations = std::stoi(text.substr(kCalculatePiCommand.length() + 1));

    // The simplest way to create a callback associated with some object is by
    // the callback factory. When a factory is created, it is bound to an object.
    // When the factory creates a callback it takes as a parameter a pointer to
    // a method in this object. When the callback factory is used in a
    // multithreaded application, developer has to ensure there won't be
    // a condition race with callback destruction. The callback can be destroyed
    // by running it or by destruction of the factory. The race can occur when
    // a callback factory is in one thread and a callback function is dispatched
    // to another thread and the factory is destroyed in the first thread.
    CompletionCallback cc = factory_.NewCallback(
        &ThreadsAndCallbacksInstance::CalculatePi, iterations);

    // The pp::SimpleThread object has a pp::MessageLoop object inside.
    // To dispatch work to a background thread we are creating a callback
    // and calling a PostWork method. The function from a callback (CalculatePi)
    // will be called in a thread other than the main PPAPI thread.
    simple_thread_pi_.message_loop().PostWork(cc);
  } else {
    if (text.find(KDownloadUrlCommand) != std::string::npos && (!loader_)) {
      DownloadUrl(text.substr(KDownloadUrlCommand.length() +1));
    }
  }
}

// The CalculatePi function uses math calculations to strain a thread.
// To show multithreaded functionalities of this demo, it is done in
// a non-main thread. For more details check a HandleMessage function.
void ThreadsAndCallbacksInstance::CalculatePi(int32_t result, int iterations) {
  std::default_random_engine generator;
  std::uniform_real_distribution<double> distribution(0.0, 1.0);

  int in_range = 0;

  for (int i = 0; i < iterations; ++i)
  {
    double x = distribution(generator);
    double y = distribution(generator);
    if (sqrt(x * x + y * y) < 1)
      ++in_range;
  }

  double pi = (in_range / static_cast<double>(iterations)) * 4;

  std::stringstream ss;
  ss << KPiPrefix << iterations << " iterations: " << pi << "\n";

  // Calls to PPAPI that don't have callback can be done from any thread.
  PostMessage(Var(ss.str()));
}

void ThreadsAndCallbacksInstance::DownloadUrl(const std::string& url) {
  // The application uses a pp::SimpleThread interface to dispatch work from
  // itself to a secondary thread. It registers a callback that will be called
  // from a background thread on this one. See demo_url_loader.h
  // The Completion callback with output provides API to return
  // some data to a callback from call.
  pp::CompletionCallbackWithOutput<std::string> cc = factory_.NewCallbackWithOutput(
      &ThreadsAndCallbacksInstance::CleanLoader);
    loader_.reset(new OwnThreadUrlLoader(InstanceHandle(pp_instance())));
    loader_->DownloadInLoaderThread(url, cc);
}

void ThreadsAndCallbacksInstance::CleanLoader(int32_t result, const std::string& message) {
  loader_.reset(nullptr);
  if (PP_OK == result)
    PostMessage(Var(message));
  else
    PostMessage(Var("Loader could not run in background thread.\n"));
}
