/**
 * Copyright (c) 2018, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @author  Mikolaj Kamionka
 * @author  Anna Bialokozowicz
 *
 */

#include "url_loader.h"

#include <algorithm>
#include <functional>
#include <string>
#include <vector>

#include "logger.h"
#include "ppapi/c/pp_completion_callback.h"
#include "ppapi/cpp/completion_callback.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/url_response_info.h"
#include "ppapi/cpp/var.h"

namespace {
  constexpr char kLocalFilePath[] = "resources/hello.txt";
  constexpr char kRemoteFileLink[] =
      "http://www.constitution.org/cons/constitu.txt";
  constexpr int64_t kBufferSize = 32768;
  constexpr int32_t kHttpOk = 200;  // 200 is HTTP status for OK.
  constexpr size_t kDisplayLines = 7;
}

FileLoaderAdapter::FileLoaderAdapter(
    const std::string& url, pp::Instance* instance,
    const std::string& file_type)
    : file_type_(file_type),
      url_(url),
      url_loader_(instance),
      instance_(instance),
      recording_progress_(false),
      callback_factory_(this) {
  loader_read_buffer_.reserve(kBufferSize);
}

LocalFileLoaderAdapter::LocalFileLoaderAdapter(pp::Instance* instance)
    : FileLoaderAdapter::FileLoaderAdapter(
        kLocalFilePath, instance, "local") {
}

RemoteFileLoaderAdapter::RemoteFileLoaderAdapter(pp::Instance* instance)
    : FileLoaderAdapter::FileLoaderAdapter(
        kRemoteFileLink, instance, "remote") {
}

void FileLoaderAdapter::RequestOpenFile(
    std::function<void()> completion_callback) {
  completion_callback_ = completion_callback;
  url_loader_.Open(SetURLRequest(url_), callback_factory_.NewCallback(
      &FileLoaderAdapter::OnURLLoaderOpenComplete));
  Logger::Log("Requested opening %s file.", file_type_.c_str());
}

bool FileLoaderAdapter::HandleResult(int32_t result) {
  // Check the result of open operation.
  if (result != PP_OK) {
    Logger::Error("Failed to open URL: %s with error code: %d",
        url_.c_str(), result);
    return false;
  }
  Logger::Log("Successfully opened URL: %s", url_.c_str());

  // Check the HTTP response code.
  pp::URLResponseInfo response_info = url_loader_.GetResponseInfo();
  if (response_info.is_null() || response_info.GetStatusCode() != kHttpOk) {
    Logger::Error("HTTP GET request failed");
    CleanAfterDownload();
    return false;
  }

  return true;
}

void RemoteFileLoaderAdapter::OnURLLoaderOpenComplete(int32_t result) {
  if (!HandleResult(result)) {
    return;
  }

  int64_t how_many_bytes_downloaded = 0;
  int64_t total_bytes = 0;

  if (url_loader_.GetDownloadProgress(&how_many_bytes_downloaded,
                                      &total_bytes)) {
    recording_progress_ = true;
  } else {
    Logger::Error("Failed to receive progress info.");
  }

  ReadDownloadedFileData();
}

void LocalFileLoaderAdapter::OnURLLoaderOpenComplete(int32_t result) {
  if (!HandleResult(result)) {
    return;
  }

  ReadDownloadedFileData();
}

void RemoteFileLoaderAdapter::ReadDownloadedFileData() {
  Logger::Log("Reading...");
  // In the contrary to the example with a local file here we are using
  // an optional callback. This type of a callback allows to receive data
  // synchronously from asynchronous functions as presented in this function.
  pp::CompletionCallback callback = callback_factory_.NewOptionalCallback(
      &RemoteFileLoaderAdapter::OnURLLoaderReadResponseBodyComplete);
  callback.set_flags(PP_COMPLETIONCALLBACK_FLAG_OPTIONAL);
  int32_t result = PP_OK;

  do {
    // ReadResponseBody function is called with an optional callback in
    // a do-while loop. If the data is ready and the loader can fill a buffer
    // immediately, it is done. Otherwise PP_OK_COMPLETIONPENDING is returned
    // and we have to wait for the callback.
    result = url_loader_.ReadResponseBody(
        loader_read_buffer_.data(), kBufferSize, callback);

    // Here we are handling synchronously received data.
    if (result < 0) {
      break;
    }

    if (recording_progress_) {
      int64_t how_many_bytes_downloaded = 0;
      int64_t total_bytes = 0;
      if (url_loader_.GetDownloadProgress(
          &how_many_bytes_downloaded, &total_bytes)) {
        ReportProgress(how_many_bytes_downloaded, total_bytes);
      }
    }
    std::move(
        loader_read_buffer_.begin(),
        loader_read_buffer_.begin() + result,
        std::back_inserter(received_data_));
  } while (true);

  if (result != PP_OK_COMPLETIONPENDING) {
    // If an error occurred or download completed,
    // we handle it in a callback function.
    callback.Run(result);
  }
}

void LocalFileLoaderAdapter::ReadDownloadedFileData() {
  Logger::Log("Reading...");
  // ReadResponseBody reads the data from URL and stores it in buffer.
  // Received data size is a kBufferSize or less if there is no more data left
  // to read. The OnURLLoaderReadResponseBodyLocalComplete function
  // will be called when read is completed.
  // If the downloaded data size is bigger than loader_read_buffer_ size,
  // ReadResponseBody need to be called until the whole data is downloaded.
  url_loader_.ReadResponseBody(loader_read_buffer_.data(), kBufferSize,
      callback_factory_.NewCallback(
          &LocalFileLoaderAdapter::OnURLLoaderReadResponseBodyComplete));
}

void FileLoaderAdapter::OnURLLoaderReadResponseBodyComplete(int32_t result) {
  // If an error occurred while trying to read data from a loader,
  // notify JavaScript and close the loader.
  if (result < 0) {
    Logger::Error("Failed to read from URL: %s", url_.c_str());
    CleanAfterDownload();
    return;
  }

  if (result == 0) {
    // If no more data to download close loader
    CleanAfterDownload();
    return;
  }

  // If the data was downloaded, the result contains the size of it in bytes.
  // Append it to the stored message and download the next fragment.
  Logger::Log("Read %d bytes", result);
  std::move(
      loader_read_buffer_.begin(),
      loader_read_buffer_.begin() + result,
      std::back_inserter(received_data_));
  ReadDownloadedFileData();
}


pp::URLRequestInfo FileLoaderAdapter::SetURLRequest(
    const std::string& url, const std::string& request_method) {
  pp::URLRequestInfo request(instance_);
  request.SetURL(url);
  request.SetMethod(request_method);
  return request;
}

pp::URLRequestInfo RemoteFileLoaderAdapter::SetURLRequest(
    const std::string& url, const std::string& request_method) {
  pp::URLRequestInfo request = FileLoaderAdapter::SetURLRequest(
      url, request_method);

  // The one, small difference in comparison with a LocalFile
  // function is that here we pass down the information to URLRequest
  // that we want to receive information about the download progress.
  request.SetRecordDownloadProgress(true);
  return request;
}

void FileLoaderAdapter::ReportProgress(
    const int64_t how_many_bytes_downloaded, const int64_t total_bytes) {
  Logger::Log("Downloaded %d of %d", how_many_bytes_downloaded, total_bytes);
}

size_t FileLoaderAdapter::CountLines(
    const std::vector<uint8_t>& received_data) {
  return std::count(received_data.begin(), received_data.end(), '\n');
}

std::vector<uint8_t> FileLoaderAdapter::GetLinesFromData(
    const std::vector<uint8_t>& received_data, const size_t line_count) {
  auto it = received_data.begin();
  auto end_bound = it;

  for (size_t i = 0; i < line_count; ++i, ++it) {
    end_bound = find(it, received_data.end(), '\n');
    if (end_bound == received_data.end())
      return received_data;
    it = end_bound;
  }
  return std::vector<uint8_t>(received_data.begin(), end_bound);
}


void FileLoaderAdapter::CleanAfterDownload() {
  recording_progress_ = false;
  std::string result_message("Received data: \n");
  size_t received_lines = CountLines(received_data_);

  if (received_lines > kDisplayLines) {
    std::vector<uint8_t> lines = GetLinesFromData(
        received_data_, kDisplayLines);
    result_message.append(lines.begin(), lines.end());
    result_message += std::to_string(received_lines - kDisplayLines);
    result_message += " lines more received.";
  } else {
    result_message.append(received_data_.begin(), received_data_.end());
  }
  Logger::Log(result_message);

  completion_callback_();
}

URLLoaderInstance::URLLoaderInstance(PP_Instance instance)
    : pp::Instance(instance) {
  file_loaders_.push_back(std::make_shared<LocalFileLoaderAdapter>(this));
  file_loaders_.push_back(std::make_shared<RemoteFileLoaderAdapter>(this));
}

bool URLLoaderInstance::Init(
    uint32_t /*argc*/, const char* /*argn*/[], const char* /*argv*/[]) {
  Logger::InitializeInstance(this);

  RunLoaderFromQueue();

  return true;
}

void URLLoaderInstance::RunLoaderFromQueue() {
  if (!file_loaders_.empty()) {
    auto loader = file_loaders_.front();
    file_loaders_.pop_front();
    loader->RequestOpenFile([loader, this](){ RunLoaderFromQueue(); });
  }
}
