/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "simple_tcp_client.h"

#include <stdio.h>
#include <sstream>
#include <string>
#include <cstring>

#include "ppapi/c/pp_errors.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "logger.h"
#include "net_utils.h"

bool SimpleTCPClient::InterfacesAreAvailable() {
  // Check if TCPSocket interface is available.
  if (!pp::TCPSocket::IsAvailable()) {
    Logger::Error("TCPSocket not available");
    return false;
  }

  // Check if HostResolver interface is available.
  if (!pp::HostResolver::IsAvailable()) {
    Logger::Error("HostResolver not available");
    return false;
  }

  return true;
}

void SimpleTCPClient::Connect(const std::string& host,
    pp::CompletionCallbackWithOutput<std::string> receive_callback) {

  // Check if this socket is already connected.
  if (IsConnected()) {
    Logger::Error("TCPClient: Already connected.");
    return;
  }

  // Create new TCPSocket which will be used for
  // the future connection.
  tcp_socket_ = pp::TCPSocket(instance_handle_);
  if (tcp_socket_.is_null()) {
    Logger::Error("TCPClient: Error creating TCPSocket.");
    return;
  }

  // Create new HostResolver to resolve IP from
  // the given host name.
  resolver_ = pp::HostResolver(instance_handle_);

  if (resolver_.is_null()) {
    Logger::Error("TCPClient: Error creating HostResolver.");
    return;
  }

  int port;
  std::string hostname;
  ParseHost(host, kPort, port, hostname);

  // We save this callback, to be called when we receive
  // a message on established connection.
  on_receive_callback_ = receive_callback;

  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPClient::OnResolveCompletion);
  PP_HostResolver_Hint hint = { PP_NETADDRESS_FAMILY_UNSPECIFIED, 0 };
  // We resolve the hostname to the IP format.
  resolver_.Resolve(hostname.c_str(), port, hint, callback);
  Logger::Log("TCPClient: Resolving ...");
}

void SimpleTCPClient::Send(const std::string& message) {
  // First we check if we are connected.
  if (!IsConnected()) {
    Logger::Error("TCPClient: Not connected.");
    return;
  }

  // We are checking if we are not already sending
  // a message.
  if (send_outstanding_) {
    Logger::Error("TCPClient: Already sending.");
    return;
  }

  uint32_t size = message.size();
  const char* data = message.c_str();
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPClient::OnSendCompletion);
  int32_t result;

  // We write a message to socket.
  result = tcp_socket_.Write(data, size, callback);
  if (result < 0) {
    if (result == PP_OK_COMPLETIONPENDING) {
      Logger::Log("TCPClient: Sending bytes: %d", size);
      send_outstanding_ = true;
    } else {
      Logger::Error("TCPClient: Send returned error: %d", result);
    }
  } else {
    Logger::Log("TCPClient: Sent bytes synchronously: %d", result);
  }
}

void SimpleTCPClient::Close() {
  // We can close only a connected socket.
  if (!IsConnected()) {
    Logger::Error("TCPClient: Not connected.");
    return;
  }

  is_connected_ = false;

  // Close the connected socket.
  tcp_socket_.Close();
  tcp_socket_ = pp::TCPSocket();

  Logger::Log("TCPClient: Closed connection.");
}

bool SimpleTCPClient::IsConnected() {
  return is_connected_;
}

void SimpleTCPClient::Receive(
    pp::CompletionCallbackWithOutput<std::string> on_receive_callback) {
  // We remember on_receive_callback callback
  // so it can be called up after receiving message.
  on_receive_callback_ = on_receive_callback;
  memset(receive_buffer_, 0, kBufferSize);
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPClient::OnReceiveCompletion);

  // We invoke reading method on the socket.
  // The receive_buffer_ variable will be filled with
  // the received message, after successful callback completion.
  tcp_socket_.Read(receive_buffer_, kBufferSize, callback);
}

void SimpleTCPClient::OnResolveCompletion(int32_t result) {
  if (result != PP_OK) {
    Logger::Error("TCPClient: Resolve failed.");
    return;
  }

  // We will use the resolved IP address from
  // the provided host name in Connect method.
  pp::NetAddress addr = resolver_.GetNetAddress(0);
  Logger::Log(std::string("TCPClient: Resolved: ") +
              addr.DescribeAsString(true).AsString());

  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleTCPClient::OnConnectCompletion);

  // We invoke Connect with the resolved IP address.
  Logger::Log("TCPClient: Connecting ...");
  tcp_socket_.Connect(addr, callback);
}

void SimpleTCPClient::OnConnectCompletion(int32_t result) {
  if (result != PP_OK) {
    Logger::Error("TCPClient: Connection failed: %d", result);
    return;
  }

  is_connected_ = true;
  Logger::Log("TCPClient: Connected");
  // Once connected, we automatically prepare for receiving
  // incoming messages.
  Receive(on_receive_callback_);
}

void SimpleTCPClient::OnReceiveCompletion(int32_t result) {
  if (result < 0) {
    Logger::Error("TCPClient: Receive failed with: %d", result);
    return;
  }

  // The receive_buffer_ variable contains our received message.
  std::string message(receive_buffer_, result);
  // We set our message as a parameter, which will be passed to
  // the callback function.
  *on_receive_callback_.output() = message;
  // We run the callback together with result.
  on_receive_callback_.Run(result);
}

void SimpleTCPClient::OnSendCompletion(int32_t result) {
  if (result < 0) {
    Logger::Error("TCPClient: Send failed with: %d", result);
  } else {
    Logger::Log("TCPClient: Sent bytes: %d", result);
  }
  send_outstanding_ = false;
}

