/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SOCKETS_SRC_SIMPLE_TCP_CLIENT_H_
#define SOCKETS_SRC_SIMPLE_TCP_CLIENT_H_

#include <string>

#include "ppapi/cpp/host_resolver.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/tcp_socket.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "config.h"

class SimpleTCPClient {
 public:
  SimpleTCPClient(pp::Instance* instance)
    : instance_handle_(instance),
      callback_factory_(this),
      on_receive_callback_(NULL),
      send_outstanding_(false),
      is_connected_(false) {
  }

  /**
   * Checks if PPAPI interfaces required for
   * this class are available.
   *
   * @return True if required interfaces are available.
   */
  static bool InterfacesAreAvailable();

  /**
   * Allows user to establish connection with remote host.
   *
   * @param [in] host Address of connection destination.
   * @param [in] receive_callback A callback used when a message was received on the socket.
   */
  void Connect(const std::string& host,
      pp::CompletionCallbackWithOutput<std::string> receive_callback);

  /**
   * Sends the message through socket.
   *
   * @param [in] message Message to be send.
   */
  void Send(const std::string& message);

  /**
   * Destroys the socket and cancels all
   * pending Send and Receive callback.
   */
  void Close();

  /**
   * Receives a message from the socket.
   *
   * @param [in] on_receive_callback Callback to function which will be called
   *             when socket will receive any data.
   */
  void Receive(
      pp::CompletionCallbackWithOutput<std::string> on_receive_callback);

 private:
  bool IsConnected();

  /**
   * Callbacks methods
   */
  void OnResolveCompletion(int32_t result);
  void OnConnectCompletion(int32_t result);
  void OnReceiveCompletion(int32_t result);
  void OnSendCompletion(int32_t result);

  pp::InstanceHandle instance_handle_;

  /**
   * A callback factory object used for easier and more convenient
   * callback creations.
   */
  pp::CompletionCallbackFactory<SimpleTCPClient> callback_factory_;

  /**
   * This callback is called after we receive information on socket.
   */
  pp::CompletionCallbackWithOutput<std::string> on_receive_callback_;

  /**
   * An object used to resolve a host name to an IP.
   */
  pp::HostResolver resolver_;

  /**
   * An object that provides a TCP socket functionality.
   */
  pp::TCPSocket tcp_socket_;

  /**
   * A buffer which is passed when receiving a message.
   */
  char receive_buffer_[kBufferSize];
  bool send_outstanding_;
  bool is_connected_;
};

#endif  // SOCKETS_SRC_SIMPLE_TCP_CLIENT_H_
