/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @brief OwnThreadUrlLoader realizes downloading of url data and returns it in
 * callback with output. Its provides functionality of downloading in loader thread.
 */

#ifndef THREADS_AND_CALLBACKS_DEMO_URL_LOADER_H_
#define THREADS_AND_CALLBACKS_DEMO_URL_LOADER_H_

#include <memory>
#include <string>

#include "ppapi/cpp/completion_callback.h"
#include "ppapi/cpp/instance_handle.h"
#include "ppapi/cpp/message_loop.h"
#include "ppapi/cpp/url_loader.h"
#include "ppapi/utility/completion_callback_factory.h"
#include "ppapi/utility/threading/simple_thread.h"

class OwnThreadUrlLoader {

 public:
  OwnThreadUrlLoader(const pp::InstanceHandle& instance);
  ~OwnThreadUrlLoader();
  // Download() function providing download in caller thread. It returns downloaded data in provided callback.
  void Download(const std::string& url, const pp::CompletionCallbackWithOutput<std::string>& callback);
  // DownloadInLoaderThread() function providing download in loader thread. It returns downloaded data in provided callback.
  void DownloadInLoaderThread(
      const std::string& url, const pp::CompletionCallbackWithOutput<std::string>& callback);

 private:
  // StartLoading() starts PPAPI URL Loader interface work
  void StartLoading(int32_t result, const std::string& url);
  // OpenCallback(), ReadResponseBodyCallback() and ReadData() are functions realizing URL downloading.
  // For more info see URL Loader Demo.
  void OpenCallback(int32_t result);
  void ReadResponseBodyCallback(int32_t result);
  void ReadData();
  // AddData() stores downloaded data in caller callback output.
  void AddData(const std::string& str);
  // SendCallback() runs callback provided by caller.
  void SendCallback(int32_t result);

  pp::InstanceHandle instance_;
  // Message loop of a caller when downloading in loader thread. Caller callback will be dispatched to this loop.
  pp::MessageLoop caller_loop_;
  // Buffer for retrieving data from PPAPI URL Loader interface and it's size.
  int64_t buffer_size_;
  std::unique_ptr<uint8_t[]> buffer_;
  // A callback provided by caller. Downloaded data is returned in output string.
  std::unique_ptr<pp::CompletionCallbackWithOutput<std::string> > caller_callback_;
  // Completion callback factory for callbacks construction.
  pp::CompletionCallbackFactory<OwnThreadUrlLoader> cc_factory_;
  // PPAPI URL Loader interface.
  pp::URLLoader loader_;
  // Simple Thread used as loader thread.
  pp::SimpleThread simple_thread_url_;
};

#endif /* THREADS_AND_CALLBACKS_DEMO_URL_LOADER_H_ */
