/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @brief
 * This is a simple hello world NaCl module based on C PPAPI interfaces.
 * It waits for a string type message form JavaScript and sends back an echo
 * message.
 * For more information about essential NaCl application structure visit:
 * @see https://developer.chrome.com/native-client/devguide/coding/application-structure
 */

#include <stdio.h>
#include <string.h>

#include "ppapi/c/pp_errors.h"
#include "ppapi/c/ppp.h"
#include "ppapi/c/ppp_instance.h"
#include "ppapi/c/ppp_messaging.h"
#include "ppapi/c/pp_bool.h"
#include "ppapi/c/ppb_var.h"
#include "ppapi/c/ppb_messaging.h"
#include "ppapi/c/pp_var.h"

/* Global pointers to selected browser interfaces. */
const PPB_Messaging* g_messaging_interface;
const PPB_Var* g_var_interface;

const char* kEcho = "Echo from NaCl: ";

/**
 * This function is called by the browser when a new NaCl application instance
 * is created. Here your application starts it's action.
 * @see <code>DidCreate</code> in ppp_instance.h file.
 */
static PP_Bool Instance_DidCreate(PP_Instance instance, uint32_t argc,
                                  const char* argn[], const char* argv[]) {
  return PP_TRUE;
}

/**
 * Called when an instance is destroyed.
 * @see <code>DidDestroy</code> in ppp_instance.h file.
 */
static void Instance_DidDestroy(PP_Instance instance) {
}

/**
 * Called when instance view attributes has changed.
 * @see <code>DidChangeView</code> in ppp_instance.h file.
 */
static void Instance_DidChangeView(PP_Instance pp_instance, PP_Resource view) {
}

/**
 * Called when an instance gained or lost focus.
 * @see <code>DidChangeFocus</code> in ppp_instance.h file.
 */
static void Instance_DidChangeFocus(PP_Instance pp_instance, PP_Bool has_focus) {
}

/**
 * @see <code>HandleDocumentLoad</code> in ppp_instance.h file.
 */
static PP_Bool Instance_HandleDocumentLoad(PP_Instance pp_instance,
                                           PP_Resource pp_url_loader) {
  return PP_FALSE;
}

/**
 * This is an entry point to this NaCl module. It's called while the module is being
 * initialized. It's better not to do much here but interface pointers initialization.
 * With <code>get_browser_interface</code> you can get browser interfaces.
 * Implementation of this method is obligatory.
 * @see <code>PPP_InitializeModule</code> in ppp.h file.
 */
PP_EXPORT int32_t PPP_InitializeModule(PP_Module module_id,
                                       PPB_GetInterface get_browser_interface) {
  // Initializing pointers to used browser interfaces.
  g_messaging_interface = (const PPB_Messaging*) get_browser_interface(
      PPB_MESSAGING_INTERFACE);
  g_var_interface = (const PPB_Var*) get_browser_interface(PPB_VAR_INTERFACE);

  return PP_OK;
}

/**
 * An implementation specific function for internal use only. It's better to
 * leave this method empty :) .
 * Implementation of this method is obligatory.
 * @see <code>PPP_ShutdownModule</code> in ppp.h file.
 */
PP_EXPORT void PPP_ShutdownModule() {
}

/**
 * Handles messages from JS sent by <code>nacl_module.postMessage(...)</code>.
 * @see <code>HandleMessage()</code> in ppp_messaging.h file.
 */
void Messaging_HandleMessage(PP_Instance instance, struct PP_Var message) {
  char str_buff[64];
  uint32_t len;

  // Extract char array from message <code>PP_Var</code> structure.
  const char* message_str = g_var_interface->VarToUtf8(message, &len);
  if (message_str == NULL)
    return;

  snprintf(str_buff, sizeof(str_buff), "%s%s\n", kEcho, message_str);

  // Create <code>PP_Var</code> containing the message body.
  struct PP_Var var_response = g_var_interface->VarFromUtf8(str_buff,
                                                            strlen(str_buff));

  // Post message to the JavaScript layer.
  g_messaging_interface->PostMessage(instance, var_response);
}

/**
 * Browser uses this method to get implementation of interfaces that this module
 * supports.
 * Implementation of this method is also obligatory.
 * @see <code>PPP_GetInterface</code> in ppp.h file.
 */
PP_EXPORT const void* PPP_GetInterface(const char* interface_name) {

  if (strcmp(interface_name, PPP_INSTANCE_INTERFACE) == 0) {
    // Your module must fill this structure (interface) with pointers to
    // previously declared functions in proper order (@see ppp.h file) and
    // return it in this function.
    static PPP_Instance instance_interface = {
        &Instance_DidCreate,
        &Instance_DidDestroy,
        &Instance_DidChangeView,
        &Instance_DidChangeFocus,
        &Instance_HandleDocumentLoad
    };
    return &instance_interface;
  }
  if (strcmp(interface_name, PPP_MESSAGING_INTERFACE) == 0) {
    // To handle messages you have to implement PPP_Messaging interface and
    // return it as a pointer to this structure.
    static PPP_Messaging messaging_interface = {
        &Messaging_HandleMessage
    };
    return &messaging_interface;
  }
  return NULL;
}

