/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @author Michal Majczak
 * @brief
 * This is a simple keyboard event handling NaCl module based on C++ PPAPI interfaces.
 * It listens to keyboard events comming from focused nacl embeded element
 * For more information about essential NaCl application structure visit:
 * @see https://developer.chrome.com/native-client/devguide/coding/application-structure
 */

#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/text_input_controller.h"
#include "ppapi/cpp/var.h"
#include "ppapi/cpp/input_event.h"

#include "logger.h"

class InputEventsInstance : public pp::Instance {
 public:
  explicit InputEventsInstance(PP_Instance instance)
      : pp::Instance(instance) {
    // Prevents showing on-screen keyboard
    pp::TextInputController text_input_controller(this);
    text_input_controller.SetTextInputType(PP_TEXTINPUT_TYPE_NONE);

    RequestInputEvents(PP_INPUTEVENT_CLASS_MOUSE | PP_INPUTEVENT_CLASS_WHEEL);
    RequestFilteringInputEvents(PP_INPUTEVENT_CLASS_KEYBOARD);
    Logger::InitializeInstance(this);
  }

  virtual ~InputEventsInstance() {
  }

  virtual void HandleMessage(const pp::Var& message) {
  }

  /**
   * This method is called when NaCl instance changes its focus state.
   * @see https://developer.chrome.com/native-client/devguide/coding/view-focus-input-events
   */
  void DidChangeFocus(bool focus) {
    if(!focus)
      Logger::Log("Lost focus!");
  }

  /**
   * This method is called when input events are captured in nacl_module instance on webpage.
   * @see https://developer.chrome.com/native-client/devguide/coding/view-focus-input-events
   */
  virtual bool HandleInputEvent(const pp::InputEvent& event) {
    switch (event.GetType()) {
      case PP_INPUTEVENT_TYPE_UNDEFINED: break;
      case PP_INPUTEVENT_TYPE_MOUSEDOWN: {
        pp::MouseInputEvent mouse_event(event);
        switch (mouse_event.GetButton()) {
            case PP_INPUTEVENT_MOUSEBUTTON_LEFT:
                Logger::Log("Left mouse button down");
              break;
            case PP_INPUTEVENT_MOUSEBUTTON_MIDDLE:
                Logger::Log("Middle mouse button down");
              break;
            case PP_INPUTEVENT_MOUSEBUTTON_RIGHT:
                Logger::Log("Right mouse button down");
              break;
            default:
              break;
          }
        } break;
      case PP_INPUTEVENT_TYPE_MOUSEUP:  {
          pp::MouseInputEvent mouse_event(event);
          switch (mouse_event.GetButton()) {
              case PP_INPUTEVENT_MOUSEBUTTON_LEFT:
                  Logger::Log("Left mouse button up");
                break;
              case PP_INPUTEVENT_MOUSEBUTTON_MIDDLE:
                  Logger::Log("Middle mouse button up");
                break;
              case PP_INPUTEVENT_MOUSEBUTTON_RIGHT:
                  Logger::Log("Right mouse button up");
                break;
              default:
                break;
            }
          } break;
      case PP_INPUTEVENT_TYPE_MOUSEMOVE: {
        pp::MouseInputEvent mouse_event(event);
        pp::Point pos = mouse_event.GetPosition();
        Logger::Log("Mouse moved to position: (%d, %d)", pos.x(), pos.y());
        } break;
      case PP_INPUTEVENT_TYPE_MOUSEENTER: break;
      case PP_INPUTEVENT_TYPE_MOUSELEAVE: break;
      case PP_INPUTEVENT_TYPE_CONTEXTMENU: break;
      case PP_INPUTEVENT_TYPE_WHEEL: {
          pp::WheelInputEvent wheel_event(event);
          pp::FloatPoint delta = wheel_event.GetDelta();
          Logger::Log("Wheel move delta: (%f, %f)", delta.x(), delta.y());
        } break;
      case PP_INPUTEVENT_TYPE_RAWKEYDOWN: break;
      case PP_INPUTEVENT_TYPE_KEYDOWN: {
        pp::KeyboardInputEvent key_event(event);
        Logger::Log("Keyboard stroke code: %u", key_event.GetKeyCode());
        } break;
      case PP_INPUTEVENT_TYPE_KEYUP: break;
      case PP_INPUTEVENT_TYPE_CHAR: {
        pp::KeyboardInputEvent key_event(event);
        pp::Var key_name = key_event.GetCharacterText();
        Logger::Log("Keyboard stroke char: \"%s\"", key_name.AsString().c_str());
        } break;
      default: break;
    }
    return true;
  }

  virtual bool Init(uint32_t argc, const char* argn[], const char* argv[]) {
    return true;
  }
};

class InputEventsModule : public pp::Module {
 public:
  InputEventsModule()
      : pp::Module() {
  }

  virtual ~InputEventsModule() {
  }

  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new InputEventsInstance(instance);
  }
};

namespace pp {

Module* CreateModule() {
  return new InputEventsModule();
}

}  // namespace pp
