/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SOCKETS_SRC_SIMPLE_UDP_SERVER_H_
#define SOCKETS_SRC_SIMPLE_UDP_SERVER_H_

#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/udp_socket.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "config.h"

struct MessageInfo {
  MessageInfo() {
  }

  MessageInfo(pp::NetAddress addr, std::string msg)
      : remoteAddress(addr),
        message(msg) {
  }

  pp::NetAddress remoteAddress;
  std::string message;
};


// Simple server based on a listening pp::UDPSocket.
// This server handles just one connection at a time and will
// run on_server_receive_from_callback_ when receiving a message
// on the listening socket.
class SimpleUDPServer {
 public:
  SimpleUDPServer(pp::Instance* instance)
      : instance_handle_(instance),
        callback_factory_(this),
        on_server_receive_from_callback_(NULL) {
  }

  /**
   * Resets active sockets. Calling this method will cause
   * to stop server to listen and close active connections.
   */
  void Reset();

  /**
   * Checks if PPAPI interfaces required for
   * this class are available.
   *
   * @return True if required interfaces are available.
   */
  static bool InterfacesAreAvailable();

  /**
   * Send a message to remote address.
   *
   * @param [in] message Message to be send.
   * @param [in] on_server_receive_from_callback_ A callback called when a new message
   *             is received.
   */
  void SendTo(const std::string& message, pp::NetAddress remote_address,
      pp::CompletionCallbackWithOutput<struct MessageInfo> on_server_receive_from_callback_);

  /**
   * Receives a message from socket.
   *
   * @param [in] port from which we will read the messages.
   * @param [in] on_server_receive_from_callback A callback called when server receives
   *             a message on the socket.
   */
  void RecvFrom(uint16_t port,
      pp::CompletionCallbackWithOutput<struct MessageInfo> on_server_receive_from_callback);

 private:
  /**
   * Callback functions
   */
  void OnBindCompletion(int32_t result, uint16_t port);
  void OnRecvFromCompletion(int32_t result, pp::NetAddress);
  void OnSendToCompletion(int32_t result);

  /*
   * Invokes reading on bound socket
   */
  void TryRead();

  pp::InstanceHandle instance_handle_;

  /**
   * Callback factory class used for easier and more convenient
   * callback creations.
   */
  pp::CompletionCallbackFactory<SimpleUDPServer> callback_factory_;

  /**
   * Holds callback for calling when we receive a message from
   * the socket.
   */
  pp::CompletionCallbackWithOutput<struct MessageInfo> on_server_receive_from_callback_;

  /**
   * Socket used as a server.
   */
  pp::UDPSocket udp_listening_socket_;

  /**
   * Buffer used for receiving message from socket.
   */
  char receive_buffer_[kBufferSize];
};

#endif  // SOCKETS_SRC_SIMPLE_UDP_SERVER_H_
