/**
 * Copyright (c) 2018, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @author  Mikolaj Kamionka
 * @author  Anna Bialokozowicz
 *
 *
 * @brief
 * This is a simple URL loader nacl module based on C++ ppapi interfaces.
 * It loads data from a local hello.txt file and an examplary remote file
 * For more information about URL Loader interface visit:
 * @see https://developer.chrome.com/native-client/devguide/coding/url-loading
 */

#ifndef URL_LOADER_SRC_URL_LOADER_H_
#define URL_LOADER_SRC_URL_LOADER_H_

#include <cstdint>
#include <deque>
#include <functional>
#include <memory>
#include <string>

#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/url_loader.h"
#include "ppapi/cpp/url_request_info.h"
#include "ppapi/utility/completion_callback_factory.h"

class FileLoaderAdapter {
 public:
  virtual ~FileLoaderAdapter() = default;
  /**
   * Opens a URL for loading from a file.
   * @param completion_callback Function to be called after completion of
   * file using
   */
  virtual void RequestOpenFile(std::function<void()> completion_callback);

 protected:
  FileLoaderAdapter(
       const std::string& url, pp::Instance* instance,
       const std::string& file_type);

  /**
   * Implements one loading step of the data
   * from loader to buffer and stores it in an internal vector.
   */
  virtual void ReadDownloadedFileData() = 0;

  /**
   * Used as a completion callback on the URLLoader::Open completion
   * in the @see RequestOpenFile
   * It starts first loading step by first call to ReadDownloadedFileData().
   * @param result The result of an asynchronus completion
   */
  virtual void OnURLLoaderOpenComplete(int32_t result) = 0;

  /**
   * Used as a completion callback on the URLLoader::ReadResponseBody completion
   * in the @see ReadDownloadedFileData()
   * it checks if download is completed, if not it starts next loading
   * step by ReadDownloadedFileData() otherwise it ends download by call to
   * CleanAfterDownload()
   * @param result The result of an asynchronus completion
   */
  void OnURLLoaderReadResponseBodyComplete(int32_t result);

  /**
   * Creates pp::URLRequestInfo and sets url and request method
   * before URLLoader::Open
   * @param url PP_URLREQUESTPROPERTY_URL property of the request
   * @param request_method PP_URLREQUESTPROPERTY_METHOD property of the request
   * @return created pp::URLRequestInfo
   */
  virtual pp::URLRequestInfo SetURLRequest(
      const std::string& url, const std::string& request_method = "GET");

  /**
   * Checks result of operation and calls CleanAfterDownload
   * in case of errors
   * @param result The result to be checked
   * @return false in case of errors, otherwise true
   */
  bool HandleResult(int32_t result);

  /**
   * Prepares and sends message with received data to
   * JavaScript and runs callback function
   */
  void CleanAfterDownload();

  /**
   * Sends information about downloading progress to JavaScript
   * @param how_many_bytes_downloaded The number of received bytes
   * @param total_bytes The total number of bytes to be received
   */
  static void ReportProgress(
      const int64_t how_many_bytes_downloaded, const int64_t total_bytes);

  /**
   * Counts lines in a vector with chars
   * @param received_data The vector in which the lines will be counted
   * @return The total number of lines in a vector
   */
  static size_t CountLines(const std::vector<uint8_t>& received_data);

  /**
   * Gets first line_count lines from a vector
   * @param received_data The vector from which the lines will be taken
   * @param line_count The number of lines to be taken from a data
   * @return The lines from a data
   */
  static std::vector<uint8_t> GetLinesFromData(
      const std::vector<uint8_t>& received_data, const size_t line_count);

  const std::string file_type_;
  const std::string url_;
  pp::URLLoader url_loader_;
  pp::Instance* instance_;
  std::vector<uint8_t> loader_read_buffer_;
  std::vector<uint8_t> received_data_;
  bool recording_progress_;
  pp::CompletionCallbackFactory<FileLoaderAdapter> callback_factory_;
  std::function<void()> completion_callback_;
};

class LocalFileLoaderAdapter : public FileLoaderAdapter {
 public:
  explicit LocalFileLoaderAdapter(pp::Instance* instance);

  ~LocalFileLoaderAdapter() override = default;

 private:
  void OnURLLoaderOpenComplete(int32_t result) override;

  void ReadDownloadedFileData() override;
};

class RemoteFileLoaderAdapter : public FileLoaderAdapter {
 public:
  explicit RemoteFileLoaderAdapter(pp::Instance* instance);

  ~RemoteFileLoaderAdapter() override = default;

 private:
  void OnURLLoaderOpenComplete(int32_t result) override;

  void ReadDownloadedFileData() override;

  pp::URLRequestInfo SetURLRequest(
      const std::string& url,
      const std::string& request_method = "GET") override;
};


class URLLoaderInstance : public pp::Instance {
 public:
  /**
   * All public methods below are part of pp::Instance class interface.
   * For more info @see Instance and hello_world_cpp demo.
   */
  explicit URLLoaderInstance(PP_Instance instance);

  ~URLLoaderInstance() override = default;

  bool Init(uint32_t argc, const char* argn[], const char* argv[]) override;

  void RunLoaderFromQueue();

 private:
  std::deque<std::shared_ptr<FileLoaderAdapter>> file_loaders_;
};

 /**
  * Standard module implementation @see Module and hello_world_cpp demo.
  */
class URLLoaderModule : public pp::Module {
 public:
  pp::Instance* CreateInstance(PP_Instance instance) override {
    return new URLLoaderInstance(instance);
  }
};

namespace pp {
  Module* CreateModule() {
    return new URLLoaderModule();
  }

}  // namespace pp

#endif // URL_LOADER_SRC_URL_LOADER_H_
