/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @file communication.js
 * @brief NaCl application communication handler. All communication should be
 * done by this file.
 */

/**
 * This function is called when a message from NaCl arrives.
 */
function handleNaclMessage(message_event) {
  var message = message_event.data;
  if (printIfLog(message)) { // function defined in common.js
    return; // this was a log or error message, so we can finish this handling
  }
  else if (typeof (message) == "string") {
      onMessage(message);
  }
}

/*
 * Communication protocol assumes, that all messages from NaCl to JS have
 * a hyphen on the beginning. Then there is a description of what is happening
 * in the NaCl module. After the description there is a number of file to which
 * the event has happened. Possible messages are as follows:
 */
// Message sent when a file is correctly loaded
var fileLoadedMessage = "-file loaded";
// Started playing
var playMessage = "-play";
// Paused playing. Playing would be resumed from the point at which it was paused.
var pauseMessage = '-pause';
// Stopped playing. Playing would start from the beginning.
var stopMessage = '-stop';
// There was an error during reading a file.
var errorMessage = '-error';
// The sound has ended and playing should be stopped.
var returnStopMessage = '-returnStop';
// The latency of playback has changed
var latencyMessage = '-latency';
// Sample frame count will be displayed
var frameMessage = '-frame';
// Sample rate was changed
var changeRadioRateMessage = '-change radio';

/**
* Handle control messages from NaCl module.
*
* @param message Message from NaCl module.
*/
function onMessage(message) {
  if (startsWith(message, fileLoadedMessage)){
    onFileLoaded(message.substr(fileLoadedMessage.length, 1));
  }
  else if (startsWith(message, playMessage)){
    onStartPlayback(message.substr(playMessage.length, 1));
  }
  else if (startsWith(message, pauseMessage)){
    onPausePlayback(message.substr(pauseMessage.length, 1));
  }
  else if (startsWith(message, stopMessage)){
    onStopPlayback(message.substr(stopMessage.length, 1));
  }
  else if (startsWith(message, errorMessage)){
    onError(message.substr(errorMessage.length, 1));
  }
  else if (startsWith(message, latencyMessage)){
    onLatency(message.substr(latencyMessage.length));
  }
  else if (startsWith(message, changeRadioRateMessage)){
    onRadioRateChange(message.substr(changeRadioRateMessage.length));
  }
  else if (startsWith(message, frameMessage)){
    onFrame(message.substr(frameMessage.length));
  }
  else if (startsWith(message, returnStopMessage)){
    var returnMessage = 'stop';
    returnMessage += message.substr(returnStopMessage.length, 1);
    document.getElementById("nacl_module").postMessage(returnMessage);
  }
  // unrecognised message
  else {
    printIfLog("ERROR: unrecognised message: " + message);
  }
}

/**
* When file data loaded, enable "play" button
*/
function onFileLoaded(number){
  document.getElementById('play' + number).disabled = false;
  document.getElementById('imgplay' + number).src = 'resources/images/play.bmp';
  document.getElementById('load' + number).disabled = true;
  document.getElementById('load' + number + 'name').disabled = true;
}

/**
* Disable "play" button, enable "stop" and "pause" buttons
*/
function onStartPlayback(number){
  console.log('Starting playback of sound ' + number);
  document.getElementById('play' + number).disabled = true;
  document.getElementById('imgplay' + number).src = 'resources/images/play-grey.bmp';
  document.getElementById('pause' + number).disabled = false;
  document.getElementById('imgpause' + number).src = 'resources/images/pause.bmp';
  document.getElementById('stop' + number).disabled = false;
  document.getElementById('imgstop' + number).src = 'resources/images/stop.bmp';
}

/**
* Enable all "play" buttons, disable "pause" button
*/
function onPausePlayback(number){
  console.log('Pausing playback of sound ' + number);
  document.getElementById('play' + number).disabled = false;
  document.getElementById('imgplay' + number).src = 'resources/images/play.bmp';
  document.getElementById('pause' + number).disabled = true;
  document.getElementById('imgpause' + number).src = 'resources/images/pause-grey.bmp';
}

/**
* Enable all "sound" and "play" buttons, disable "stop" button
*/
function onStopPlayback(number){
  console.log('Stopping playback of sound ' + number);
  document.getElementById('play' + number).disabled = false;
  document.getElementById('imgplay' + number).src = 'resources/images/play.bmp';
  document.getElementById('pause' + number).disabled = true;
  document.getElementById('imgpause' + number).src = 'resources/images/pause-grey.bmp';
  document.getElementById('stop' + number).disabled = true;
  document.getElementById('imgstop' + number).src = 'resources/images/stop-grey.bmp';
}

function onError(number){
  var fileName = document.getElementById('load' + number + 'name').value;
  printIfLog("ERROR: Can't open file: " + fileName + '\n');
}

/**
* Display latency for presentation of provided audio samples
*/
function onLatency(latency){
  document.getElementById('latency').value = latency;
}

/**
* When Sample Rate Changes proper radio is checked
*/
function onRadioRateChange(rate){
  document.getElementById(rate).checked = true;
}
/**
* Display sample frame count for presentation of provided audio samples
*/
function onFrame(frame){
  document.getElementById('frame').value = frame;
}
