/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @author  Tomasz Borkowski
 *
 * @brief
 * This is a NaCl Logger that sends text messages (and adds a specific prefix)
 * to JS side with use of pp::Instance.PostMessage() method.
 */

#ifndef COMMON_SRC_LOGGER_H_
#define COMMON_SRC_LOGGER_H_

#include <stdarg.h>
#include <string>

#include "ppapi/cpp/instance.h"

/**
 * Utility class that simplifies sending log messages by PostMessage to JS.
 */
class Logger {
 public:
  /**
   * Initializes the pp::Instance pointer, so that the Logger could post
   * messages to JS.
   */
  static void InitializeInstance(pp::Instance* instance);

  /**
   * Adds a log prefix and a newline character at the end to the passed string
   * and sends it to JS.
   */
  static void Log(const std::string& message);

  /**
   * Does the same as log(const std::string&), but takes arguments like standard
   * stdio printf() function.
   * \link http://www.cplusplus.com/reference/cstdio/printf/
   */
  static void Log(const char* message_format, ...);

  /**
   * Does the same as log( ), but adds error prefix.
   */
  static void Error(const std::string& message);
  static void Error(const char* message_format, ...);

  /**
   * Similar to Error and Log, but with debug prefix. These logs are disabled
   * by default. <code>EnableDebugLogs(true)</code> must be called to show these
   * logs.
   */
  static void Debug(const std::string& message);
  static void Debug(const char* message_format, ...);

  /**
   * Enables logs sent by <code>Debug</code> methods if passed <code>flag</code>
   * is true. Disables debug logs when <code>flag</code> is false.
   * By default debug logs are disabled.
   */
  static void EnableDebugLogs(bool flag);

 private:
  /**
   * Internal wrappers for printing to PostMessage with specified prefix.
   */
  static void InternalPrint(const char* prefix, const std::string& message);
  static void InternalPrint(const char* prefix, const char* message_format,
                            va_list arguments_list);

  static pp::Instance* instance_;
  static bool show_debug_;
};

#endif /* COMMON_SRC_LOGGER_H_ */
