/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "graphics_3d.h"

#include <math.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <GLES2/gl2.h>

#include "ppapi/cpp/graphics_3d.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/text_input_controller.h"
#include "ppapi/cpp/var.h"
#include "ppapi/cpp/var_array.h"
#include "ppapi/lib/gl/gles2/gl2ext_ppapi.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "logger.h"
#include "matrix.h"

// Pointer to externally initialized texture data in BGR format.
extern const uint8_t* kTextureData;

namespace {

// Camera frustum parameters.
const float kFovY = 45.0f;
const float kZNear = 1.0f;
const float kZFar = 10.0f;
const float kCameraZ = -4.0f;
const float kXAngleDelta = 2.0f;
const float kYAngleDelta = 0.5f;

/**
 * Compiles a single shader program.
 */
GLuint CompileShader(GLenum type, const char* data) {
  GLuint shader = glCreateShader(type);
  glShaderSource(shader, 1, &data, NULL);
  glCompileShader(shader);

  GLint compile_status;
  glGetShaderiv(shader, GL_COMPILE_STATUS, &compile_status);
  if (compile_status != GL_TRUE) {
    // Shader failed to compile, let's see what the error is.
    char buffer[1024];
    GLsizei length;
    glGetShaderInfoLog(shader, sizeof(buffer), &length, &buffer[0]);
    Logger::Error("Shader failed to compile: %s", buffer);
    return 0;
  }

  return shader;
}

/**
 * Links compiled shader programs.
 */
GLuint LinkProgram(GLuint frag_shader, GLuint vert_shader) {
  GLuint program = glCreateProgram();
  glAttachShader(program, frag_shader);
  glAttachShader(program, vert_shader);
  glLinkProgram(program);

  GLint link_status;
  glGetProgramiv(program, GL_LINK_STATUS, &link_status);
  if (link_status != GL_TRUE) {
    // Program failed to link, let's see what the error is.
    char buffer[1024];
    GLsizei length;
    glGetProgramInfoLog(program, sizeof(buffer), &length, &buffer[0]);
    Logger::Error("Program failed to link: %s", buffer);
    return 0;
  }

  return program;
}

/**
 * Fragment shader source code.
 * This shader is responsible of applying proper color to the rasterized objects
 * from Vertex shader taking texture colors into account.
 */
const char kFragShaderSource[] =
    "precision mediump float;\n"
    "varying vec3 v_color;\n"
    "varying vec2 v_texcoord;\n"
    "uniform sampler2D u_texture;\n"
    "void main() {\n"
    "  gl_FragColor = texture2D(u_texture, v_texcoord);\n"
    "  gl_FragColor += vec4(v_color, 1);\n"
    "}\n";

/**
 * Vertex shader source code.
 * The only thing this shader does is transforming all vertices by provided
 * transformation matrix and then rasterizing them to be drawn on the buffer.
 */
const char kVertexShaderSource[] =
    "uniform mat4 u_mvp;\n"
    "attribute vec2 a_texcoord;\n"
    "attribute vec3 a_color;\n"
    "attribute vec4 a_position;\n"
    "varying vec3 v_color;\n"
    "varying vec2 v_texcoord;\n"
    "void main() {\n"
    "  gl_Position = u_mvp * a_position;\n"
    "  v_color = a_color;\n"
    "  v_texcoord = a_texcoord;\n"
    "}\n";

// Structure representing Vertex data: location, base color and texture uv
// coordinates.
struct Vertex {
  float loc[3];
  float color[3];
  float tex[2];
};

// Array containing all vertex data of the drawn cube, each face is described
// by 4 vertices.
const Vertex kCubeVerts[24] = {
  // +Z (red arrow, black tip)
  {{-1.0, -1.0, +1.0}, {0.0, 0.0, 0.0}, {1.0, 0.0}},
  {{+1.0, -1.0, +1.0}, {0.0, 0.0, 0.0}, {0.0, 0.0}},
  {{+1.0, +1.0, +1.0}, {0.5, 0.0, 0.0}, {0.0, 1.0}},
  {{-1.0, +1.0, +1.0}, {0.5, 0.0, 0.0}, {1.0, 1.0}},

  // +X (green arrow, black tip)
  {{+1.0, -1.0, -1.0}, {0.0, 0.0, 0.0}, {1.0, 0.0}},
  {{+1.0, +1.0, -1.0}, {0.0, 0.0, 0.0}, {0.0, 0.0}},
  {{+1.0, +1.0, +1.0}, {0.0, 0.5, 0.0}, {0.0, 1.0}},
  {{+1.0, -1.0, +1.0}, {0.0, 0.5, 0.0}, {1.0, 1.0}},

  // +Y (blue arrow, black tip)
  {{-1.0, +1.0, -1.0}, {0.0, 0.0, 0.0}, {1.0, 0.0}},
  {{-1.0, +1.0, +1.0}, {0.0, 0.0, 0.0}, {0.0, 0.0}},
  {{+1.0, +1.0, +1.0}, {0.0, 0.0, 0.5}, {0.0, 1.0}},
  {{+1.0, +1.0, -1.0}, {0.0, 0.0, 0.5}, {1.0, 1.0}},

  // -Z (red arrow, red tip)
  {{+1.0, +1.0, -1.0}, {0.0, 0.0, 0.0}, {1.0, 1.0}},
  {{-1.0, +1.0, -1.0}, {0.0, 0.0, 0.0}, {0.0, 1.0}},
  {{-1.0, -1.0, -1.0}, {1.0, 0.0, 0.0}, {0.0, 0.0}},
  {{+1.0, -1.0, -1.0}, {1.0, 0.0, 0.0}, {1.0, 0.0}},

  // -X (green arrow, green tip)
  {{-1.0, +1.0, +1.0}, {0.0, 0.0, 0.0}, {1.0, 1.0}},
  {{-1.0, -1.0, +1.0}, {0.0, 0.0, 0.0}, {0.0, 1.0}},
  {{-1.0, -1.0, -1.0}, {0.0, 1.0, 0.0}, {0.0, 0.0}},
  {{-1.0, +1.0, -1.0}, {0.0, 1.0, 0.0}, {1.0, 0.0}},

  // -Y (blue arrow, blue tip)
  {{+1.0, -1.0, +1.0}, {0.0, 0.0, 0.0}, {1.0, 1.0}},
  {{+1.0, -1.0, -1.0}, {0.0, 0.0, 0.0}, {0.0, 1.0}},
  {{-1.0, -1.0, -1.0}, {0.0, 0.0, 1.0}, {0.0, 0.0}},
  {{-1.0, -1.0, +1.0}, {0.0, 0.0, 1.0}, {1.0, 0.0}},
};

// Indices drawing order, each triplet describes one triangle to be drawn.
const GLubyte kCubeIndexes[36] = {
   2,  1,  0,  3,  2,  0,
   6,  5,  4,  7,  6,  4,
  10,  9,  8, 11, 10,  8,
  14, 13, 12, 15, 14, 12,
  18, 17, 16, 19, 18, 16,
  22, 21, 20, 23, 22, 20,
};

}  // namespace


Graphics3DCubeInstance::Graphics3DCubeInstance(PP_Instance instance)
    : pp::Instance(instance),
      callback_factory_(this),
      width_(0),
      height_(0),
      device_scale_(0.0f),
      frag_shader_(0),
      vertex_shader_(0),
      program_(0),
      vertex_buffer_(0),
      index_buffer_(0),
      texture_(0),
      texture_loc_(0),
      position_loc_(0),
      texcoord_loc_(0),
      color_loc_(0),
      mvp_loc_(0),
      mouse_down_(false),
      mouse_moved_on_click_(false),
      x_angle_(0),
      y_angle_(0),
      animating_(true) {
}

bool Graphics3DCubeInstance::Init(uint32_t argc, const char* argn[],
                                  const char* argv[]) {
  // Prevents showing on-screen keyboard
  pp::TextInputController text_input_controller(this);
  text_input_controller.SetTextInputType(PP_TEXTINPUT_TYPE_NONE);

  RequestInputEvents(PP_INPUTEVENT_CLASS_MOUSE);
  Logger::InitializeInstance(this);
  return true;
}

void Graphics3DCubeInstance::DidChangeView(const pp::View& view) {
  // Pepper specifies dimensions in DIPs (device-independent pixels). To
  // generate a context that is at device-pixel resolution on HiDPI devices,
  // scale the dimensions by view.GetDeviceScale().
  device_scale_ = view.GetDeviceScale();
  int32_t new_width = view.GetRect().width() * device_scale_;
  int32_t new_height = view.GetRect().height() * device_scale_;

  if (context_.is_null()) {
    if (!InitGL(new_width, new_height)) {
      Logger::Error("Couldn't initialize GLES library!");
      return;
    }
    // Everything is ok, prepare pipeline and start drawing.
    InitShaders();
    InitBuffers();
    InitTexture();
    MainLoopIteration(0);
  } else {
    // Resize the buffers to the new size of the module.
    int32_t result = context_.ResizeBuffers(new_width, new_height);
    if (result < 0) {
      Logger::Error("Unable to resize buffers to %d x %d!", new_width,
                    new_height);
      return;
    }
  }

  width_ = new_width;
  height_ = new_height;
  glViewport(0, 0, width_, height_);
  Logger::Log("Initialized module's view with resolution %dx%d",
              width_, height_);
}

bool Graphics3DCubeInstance::HandleInputEvent(const pp::InputEvent& event) {
  pp::MouseInputEvent mouse_event(event);
  if (mouse_event.is_null())
    return true;

  switch (mouse_event.GetType()) {
    case PP_INPUTEVENT_TYPE_MOUSEDOWN: {
      mouse_down_ = true;
      break;
    }
    case PP_INPUTEVENT_TYPE_MOUSEMOVE: {
      pp::Point tmp_point = pp::Point(
          mouse_event.GetPosition().x() * device_scale_,
          mouse_event.GetPosition().y() * device_scale_);
      // Stop animation if user moves the mouse with a pressed button.
      if (mouse_down_) {
        if (tmp_point != current_mouse_point_)
          mouse_moved_on_click_ = true;
        animating_ = false;
      }
      current_mouse_point_ = tmp_point;
      break;
    }
    case PP_INPUTEVENT_TYPE_MOUSEUP: {
      // Mouse click with no move toggles the cube animation.
      if (!mouse_moved_on_click_) {
        animating_ = !animating_;
      }
      mouse_moved_on_click_ = false;
      mouse_down_ = false;
      break;
    }
    default:
      break;
  }

  return true;
}

bool Graphics3DCubeInstance::InitGL(int32_t new_width, int32_t new_height) {
  // Initialize OpenGL ES library and its connection with this NaCl module.
  // This must be called once before making any gl calls.
  // @see ppapi/lib/gl/gles2/gl2ext_ppapi.h
  if (!glInitializePPAPI(pp::Module::Get()->get_browser_interface())) {
    Logger::Error("Unable to initialize GLES PPAPI!");
    return false;
  }

  const int32_t attrib_list[] = {
    PP_GRAPHICS3DATTRIB_ALPHA_SIZE, 8,
    PP_GRAPHICS3DATTRIB_DEPTH_SIZE, 24,
    PP_GRAPHICS3DATTRIB_WIDTH, new_width,
    PP_GRAPHICS3DATTRIB_HEIGHT, new_height,
    PP_GRAPHICS3DATTRIB_NONE
  };

  // Create a 3D graphics context and bind it with this <code>Instance</code>.
  context_ = pp::Graphics3D(this, attrib_list);
  if (!BindGraphics(context_)) {
    Logger::Error("Unable to bind 3D context!");
    context_ = pp::Graphics3D();
    glSetCurrentContextPPAPI(0);
    return false;
  }

  // Set context to be used for rendering.
  glSetCurrentContextPPAPI(context_.pp_resource());
  Logger::Log("Initialized GLES library.");
  return true;
}

void Graphics3DCubeInstance::InitShaders() {
  frag_shader_ = CompileShader(GL_FRAGMENT_SHADER, kFragShaderSource);
  if (!frag_shader_)
    return;

  vertex_shader_ = CompileShader(GL_VERTEX_SHADER, kVertexShaderSource);
  if (!vertex_shader_)
    return;

  program_ = LinkProgram(frag_shader_, vertex_shader_);
  if (!program_)
    return;

  // Save uniforms and attributes location for future usage.
  texture_loc_ = glGetUniformLocation(program_, "u_texture");
  position_loc_ = glGetAttribLocation(program_, "a_position");
  texcoord_loc_ = glGetAttribLocation(program_, "a_texcoord");
  color_loc_ = glGetAttribLocation(program_, "a_color");
  mvp_loc_ = glGetUniformLocation(program_, "u_mvp");
}

void Graphics3DCubeInstance::InitBuffers() {
  glGenBuffers(1, &vertex_buffer_);
  glBindBuffer(GL_ARRAY_BUFFER, vertex_buffer_);
  glBufferData(GL_ARRAY_BUFFER, sizeof(kCubeVerts), &kCubeVerts[0],
               GL_STATIC_DRAW);

  glGenBuffers(1, &index_buffer_);
  glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, index_buffer_);
  glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(kCubeIndexes),
               &kCubeIndexes[0], GL_STATIC_DRAW);
}

void Graphics3DCubeInstance::InitTexture() {
  glGenTextures(1, &texture_);
  glBindTexture(GL_TEXTURE_2D, texture_);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 128, 128, 0, GL_RGB, GL_UNSIGNED_BYTE,
               &kTextureData[0]);
}

void Graphics3DCubeInstance::Animate() {
  // Follow the mouse movement if a mouse button is down.
  if (mouse_down_) {
    pp::Point position_delta = last_mouse_point_ - current_mouse_point_;
    // X and Y axes in 3D are directed differently than 2D mouse position axes.
    y_angle_ += position_delta.x() / static_cast<float>(width_) * 360.0f;
    x_angle_ -= position_delta.y() / static_cast<float>(width_) * 360.0f;
  }
  last_mouse_point_ = current_mouse_point_;

  if (animating_) {
    x_angle_ += kXAngleDelta;
    y_angle_ += kYAngleDelta;
  }

  // Normalize to range <0.0, 360.0> degrees.
  x_angle_ = fmod(360.0f + x_angle_, 360.0f);
  y_angle_ = fmod(360.0f + y_angle_, 360.0f);
}

void Graphics3DCubeInstance::Render() {
  // Clear current buffer.
  glClearColor(0.5, 0.5, 0.5, 1);
  glClearDepthf(1.0f);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glEnable(GL_DEPTH_TEST);

  // Set what program to use.
  glUseProgram(program_);
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, texture_);
  glUniform1i(texture_loc_, 0);

  // Create our perspective matrix.
  float mvp[16];
  float trs[16];
  float rot[16];

  identity_matrix(mvp);
  const float aspect_ratio = static_cast<float>(width_) / height_;
  glhPerspectivef2(&mvp[0], kFovY, aspect_ratio, kZNear, kZFar);
  // Prepare transformation matrix.
  translate_matrix(0, 0, kCameraZ, trs);
  rotate_matrix(x_angle_, y_angle_, 0.0f, rot);
  multiply_matrix(trs, rot, trs);
  multiply_matrix(mvp, trs, mvp);
  glUniformMatrix4fv(mvp_loc_, 1, GL_FALSE, mvp);

  // Define the attributes of the vertex.
  // Each attribute has information on it's specific data offset in the array,
  // to be able to iterate over it.
  glBindBuffer(GL_ARRAY_BUFFER, vertex_buffer_);
  glVertexAttribPointer(position_loc_, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex),
                        reinterpret_cast<void*>(offsetof(Vertex, loc)));
  glEnableVertexAttribArray(position_loc_);
  glVertexAttribPointer(color_loc_, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex),
                        reinterpret_cast<void*>(offsetof(Vertex, color)));
  glEnableVertexAttribArray(color_loc_);
  glVertexAttribPointer(texcoord_loc_, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex),
                        reinterpret_cast<void*>(offsetof(Vertex, tex)));
  glEnableVertexAttribArray(texcoord_loc_);

  // Bind buffer containing indicies drawing order and draw all bound elements
  // to the background buffer.
  glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, index_buffer_);
  glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_BYTE, 0);
}

void Graphics3DCubeInstance::MainLoopIteration(int32_t) {
  Animate();
  Render();
  // Swap the background buffer with the foreground buffer.
  context_.SwapBuffers(
      callback_factory_.NewCallback(&Graphics3DCubeInstance::MainLoopIteration));
}

