/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef GRAPHICS_3D_SRC_GRAPHICS_3D_H_
#define GRAPHICS_3D_SRC_GRAPHICS_3D_H_

#include <GLES2/gl2.h>

#include "ppapi/cpp/graphics_3d.h"
#include "ppapi/cpp/input_event.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"

class Graphics3DCubeInstance : public pp::Instance {
 public:
  Graphics3DCubeInstance(PP_Instance instance);
  virtual bool Init(uint32_t argc, const char* argn[], const char* argv[]);
  virtual bool HandleInputEvent(const pp::InputEvent& event);

  /**
   * Called on viewport change and on viewport initialization.
   */
  virtual void DidChangeView(const pp::View& view);

 private:
  /**
   *  Initialize Graphics Library. Most of PPAPI specific code associated with
   *  3D graphics can be found in this method.
   */
  bool InitGL(int32_t new_width, int32_t new_height);

  /**
   * Prepares shader program.
   */
  void InitShaders();

  /**
   * Creates and populates buffers with cube vertices data.
   */
  void InitBuffers();

  /**
   * Initializes texture params and loads it to GPU.
   */
  void InitTexture();

  /**
   * Recalculates transformation matrix according to the current object rotation,
   * it's called for every frame.
   */
  void Animate();

  /**
   * Rendering method that is called for every frame.
   */
  void Render();

  /**
   * Called once per frame.
   * The <code>int32_t</code> parameter is not used. It's needed to fit a
   * CompletionCallback function signature.
   */
  void MainLoopIteration(int32_t);

  pp::CompletionCallbackFactory<Graphics3DCubeInstance> callback_factory_;
  pp::Graphics3D context_;
  int32_t width_;
  int32_t height_;
  float device_scale_;
  GLuint frag_shader_;
  GLuint vertex_shader_;
  GLuint program_;
  GLuint vertex_buffer_;
  GLuint index_buffer_;
  GLuint texture_;

  GLuint texture_loc_;
  GLuint position_loc_;
  GLuint texcoord_loc_;
  GLuint color_loc_;
  GLuint mvp_loc_;

  bool mouse_down_;
  bool mouse_moved_on_click_;
  pp::Point current_mouse_point_;
  pp::Point last_mouse_point_;

  float x_angle_;
  float y_angle_;
  bool animating_;
};

class Graphics3DCubeModule : public pp::Module {
 public:
  Graphics3DCubeModule() : pp::Module() {
  }

  virtual ~Graphics3DCubeModule() {
  }

  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new Graphics3DCubeInstance(instance);
  }
};

namespace pp {
Module* CreateModule() {
  return new Graphics3DCubeModule();
}
}  // namespace pp

#endif /* GRAPHICS_3D_SRC_GRAPHICS_3D_H_ */
