/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "simple_udp_server.h"

#include <stdio.h>
#include <sstream>
#include <string>

#include "ppapi/c/pp_errors.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "logger.h"
#include "net_utils.h"

void SimpleUDPServer::Reset() {
  udp_listening_socket_ = pp::UDPSocket(instance_handle_);
}

bool SimpleUDPServer::InterfacesAreAvailable() {
  // Check if UDPSocket interface is available.
  if (!pp::UDPSocket::IsAvailable()) {
    Logger::Error("UDPServer: UDPSocket not available");
    return false;
  }

  return true;
}

void SimpleUDPServer::RecvFrom(uint16_t port,
    pp::CompletionCallbackWithOutput<struct MessageInfo> on_server_receive_from_callback) {
  udp_listening_socket_ = pp::UDPSocket(instance_handle_);
  if (udp_listening_socket_.is_null()) {
    Logger::Error("UDPServer: Error creating UDPSocket.");
    return;
  }

  // Save the receive callback for later use.
  on_server_receive_from_callback_ = on_server_receive_from_callback;
  // Attempt to listen on all interfaces (0.0.0.0)
  // on the given port number.
  PP_NetAddress_IPv4 ipv4_addr = { Htons(port), { 0 } };
  pp::NetAddress addr(instance_handle_, ipv4_addr);
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleUDPServer::OnBindCompletion, port);

  // We are binding to the specific address.
  int32_t rtn = udp_listening_socket_.Bind(addr, callback);
  if (rtn != PP_OK_COMPLETIONPENDING) {
    Logger::Error("UDPServer: Error binding listening socket.");
    return;
  }
}

void SimpleUDPServer::OnBindCompletion(int32_t result, uint16_t port) {
  if (result != PP_OK) {
    Logger::Error("UDPServer: server: Bind failed with: %d", result);
    return;
  }

  Logger::Log("UDPServer: Listening on port: %d", port);

  pp::CompletionCallbackWithOutput<pp::NetAddress> callback =
      callback_factory_.NewCallbackWithOutput(
          &SimpleUDPServer::OnRecvFromCompletion);

  // We start receiving data on the socket.
  udp_listening_socket_.RecvFrom(receive_buffer_, kBufferSize, callback);
}

void SimpleUDPServer::SendTo(const std::string& message, pp::NetAddress remote_address,
    pp::CompletionCallbackWithOutput<struct MessageInfo> on_server_receive_from_callback) {

  // Save the receive callback for later use.
  on_server_receive_from_callback_ = on_server_receive_from_callback;
  uint32_t size = message.size();
  const char* data = message.c_str();
  pp::CompletionCallback callback =
      callback_factory_.NewCallback(&SimpleUDPServer::OnSendToCompletion);
  int32_t result;

  // Write a message to the socket.
  result = udp_listening_socket_.SendTo(data, size, remote_address, callback);
  if (result < 0) {
    if (result == PP_OK_COMPLETIONPENDING) {
      Logger::Log("UDPServer: Sending bytes: %d", size);
    } else {
      Logger::Error("UDPServer: Send returned error: %d", result);
    }
  } else {
    Logger::Log("UDPServer: Sent bytes synchronously: %d", result);
  }
}

void SimpleUDPServer::OnRecvFromCompletion(int32_t result,
                                           pp::NetAddress address) {
  if (result <= 0) {
    if(result == PP_ERROR_ABORTED) {
      // We consume aborted callback. This can occur when current server
      // is being shutdown.
      Logger::Log("UDPServer: Consuming old RecvFrom callback");
      return;
    }
    if (result == 0)
      Logger::Error("UDPServer: client disconnected");
    else
      Logger::Error("UDPServer: Read failed: %d", result);
    on_server_receive_from_callback_.Run(result);
    return;
  }
  std::string message(receive_buffer_, result);
  struct MessageInfo ext_message(address, message);
  // We set our message as a parameter, which will be
  // passed to the callback function.
  *on_server_receive_from_callback_.output() = ext_message;
  on_server_receive_from_callback_.Run(result);
}

void SimpleUDPServer::OnSendToCompletion(int32_t result) {
  if (result < 0) {
    Logger::Error("UDPServer: Send failed: %d", result);
    return;
  }

  pp::CompletionCallbackWithOutput<pp::NetAddress> callback =
      callback_factory_.NewCallbackWithOutput(
          &SimpleUDPServer::OnRecvFromCompletion);
  // Try to read bytes from the socket.
  udp_listening_socket_.RecvFrom(receive_buffer_, kBufferSize, callback);
}
