/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "sockets.h"

#include "logger.h"

bool SocketsInstance::Init(uint32_t argc, const char* argn[],
                           const char* argv[]) {
  UNUSED(argc);
  UNUSED(argn);
  UNUSED(argv);

  Logger::InitializeInstance(this);

  if (!SimpleTCPClient::InterfacesAreAvailable()) {
    Logger::Error("SocketsInstance: Failed to initialize SimpleTCPClient");
    return false;
  }

  // This check is redundant, because we are already
  // checking if TCPSocket interface is available in method
  // SimpleTCPClient::InterfacesAreAvailable().
  if (!SimpleTCPServer::InterfacesAreAvailable()) {
    Logger::Error("SocketsInstance: Failed to initialize SimpleTCPServer");
    return false;
  }

  if (!SimpleUDPClient::InterfacesAreAvailable()) {
    Logger::Error("SocketsInstance: Failed to initialize SimpleUDPClient");
    return false;
  }

  if (!SimpleUDPServer::InterfacesAreAvailable()) {
    Logger::Error("SocketsInstance: Failed to initialize SimpleUDPServer");
    return false;
  }

  return true;
}

void SocketsInstance::HandleMessage(const pp::Var& var_message) {
  if (!var_message.is_string())
    return;
  std::string message = var_message.AsString();
  // This message must contain a command character followed by ';' with
  // optional subcommand and arguments like "X;arguments" or "X;X;arguments".
  if (message.length() < 2 || message[1] != ';')
    return;
  switch (message[0]) {
    case MSG_UDP: {
      // The command 'u' requests to create a UDP connection to the
      // specified HOST.
      // HOST is passed as an argument like "u;HOST".
      pp::CompletionCallbackWithOutput<std::string> on_client_receive_callback =
          callback_factory_.NewCallbackWithOutput(
              &SocketsInstance::OnUDPClientReceiveFromCallback);
      udp_client_.Bind(message.substr(2), on_client_receive_callback);
      break;
    }
    case MSG_TCP: {
      // The command 't' requests to connect to the specified HOST.
      // HOST is passed as an argument like "t;HOST".
      pp::CompletionCallbackWithOutput<std::string> on_client_receive_callback =
          callback_factory_.NewCallbackWithOutput(
              &SocketsInstance::OnTCPClientReceiveCallback);
      tcp_client_.Connect(message.substr(2), on_client_receive_callback);
      break;
    }
    case MSG_CLOSE:
      // The command 'c' requests to close current socket. The
      // sub-command specifies type of socket ('t' == tcp,
      // 'u' == udp). Message can look like 'c;t;' or 'c;u;'.
      if (message[2] == MSG_TCP)
        tcp_client_.Close();
      else
        udp_client_.Close();
      break;
    case MSG_LISTEN: {
      // The command 'l' starts a listening socket (server).
      // The sub-command specifies type of socket ('t' == tcp,
      // 'u' == udp). Message can look like 'l;t;' or 'l;u;'.
      int port = atoi(message.substr(4).c_str());
      if (message[2] == MSG_TCP) {
        current_tcp_port_ = port;
        StartListenOnTCPSocket(port);
      } else {
        // If previous server existed, then after this call it will close
        // all existing connections on it.
        udp_server_.Reset();
        pp::CompletionCallbackWithOutput<struct MessageInfo> on_server_receive_callback =
            callback_factory_.NewCallbackWithOutput(
                &SocketsInstance::OnUDPServerReceiveCallback);

        // We start waiting for incoming messages.
        udp_server_.RecvFrom(port, on_server_receive_callback);
      }
      break;
    }
    case MSG_SEND:
      // The command 's' requests to send a message as a text frame. The
      // sub-command specifies if socket is TCP ('t' sub-command) or UDP
      // ('u' sub-command). The message passed as an argument like
      // "s;t;message" or "s;u;message".
      if (message[2] == MSG_TCP)
        tcp_client_.Send(message.substr(4));
      else
        udp_client_.SendTo(message.substr(4));
      break;
    default:
      Logger::Error("SocketsInstance: Unhandled message from JavaScript: %s", message.c_str());
      break;
  }
}

// This callback will be run when we receive incoming message on
// client socket.
void SocketsInstance::OnTCPClientReceiveCallback(int32_t result,
                                                 const std::string& message) {
  if (result <= 0) {
    Logger::Error("SocketsInstance: TCPClient receive failed with: %d", result);
    return;
  }

  Logger::Log("SocketsInstance: TCPClient received: %s", message.c_str());

  tcp_client_.Receive(
      callback_factory_.NewCallbackWithOutput(
          &SocketsInstance::OnTCPClientReceiveCallback));
}

// This callback will be run when we receive incoming message on
// server socket.
void SocketsInstance::OnTCPServerReceiveCallback(int32_t result,
                                                 const std::string& message) {
  if (result <= 0) {
      if (result == 0) {
        Logger::Error("SocketsInstance: TCPServer client disconnected");
        // Client have disconnected from server, thus start
        // listen for other incoming clients.
        StartListenOnTCPSocket(current_tcp_port_);
      } else {
        Logger::Error("TCPServer: Read failed: %d", result);
      }
      return;
  }

  Logger::Log("SocketsInstance: TCPServer received: %s\n\t  Sending back the "
              "same message (echo).", message.c_str());
  // We respond to sender with message which we received from him
  tcp_server_.Write(message,
                    callback_factory_.NewCallbackWithOutput(
                        &SocketsInstance::OnTCPServerReceiveCallback));
}

// This callback will be run upon incoming new connection.
void SocketsInstance::OnTCPServerAcceptCallback(int32_t result,
                                                const std::string& ip) {
  if (result < 0) {
    Logger::Error("SocketsInstance: Accept failed: %d", result);
    return;
  }

  Logger::Log("SocketsInstance: TCPServer have accepted connection from %s",
              ip.c_str());
}

// This callback will be run when we receive incoming message on
// client socket.
void SocketsInstance::OnUDPClientReceiveFromCallback(int32_t result,
                                                     const std::string& message) {
  if (result <= 0) {
    Logger::Error("SocketsInstance: Client ReceiveFrom failed: %d", result);
    return;
  }

  Logger::Log("SocketsInstance: UDPClient received: %s \n\t  Sending back the same "
              "message (echo).", message.c_str());
  // After receiving message, we invoke another Receive method to
  // listen for another future incoming messages.
  udp_client_.Receive(
      callback_factory_.NewCallbackWithOutput(
          &SocketsInstance::OnUDPClientReceiveFromCallback));
}

// This callback will be run when we receive incoming message on
// server socket.
void SocketsInstance::OnUDPServerReceiveCallback(
    int32_t result, struct MessageInfo extended_message) {

  if (result <= 0) {
    Logger::Error("SocketsInstance: Server ReceiveFrom failed: %d", result);
    return;
  }
  // Below we are doing simple conversion to char[20] to print
  // IP of host which sended message.
  PP_NetAddress_IPv4 ipv4;
  extended_message.remoteAddress.DescribeAsIPv4Address(&ipv4);
  char buff[20];
  snprintf(buff, sizeof(buff), "%d.%d.%d.%d:%d", ipv4.addr[0], ipv4.addr[1],
           ipv4.addr[2], ipv4.addr[3], ipv4.port);
  Logger::Log("SocketsInstance: UDPServer Host: %s -> Sends : %s\n\t  Sending "
      "back the same message (echo).", buff, extended_message.message.c_str());
  // We respond to a sender with a message which we received from him
  udp_server_.SendTo(extended_message.message, extended_message.remoteAddress,
                     callback_factory_.NewCallbackWithOutput(
                         &SocketsInstance::OnUDPServerReceiveCallback));
}

void SocketsInstance::StartListenOnTCPSocket(int port) {
  // Create callbacks which will be called upon accepting
  // of incoming connection and receiving message.
  pp::CompletionCallbackWithOutput<std::string> on_server_receive_callback =
      callback_factory_.NewCallbackWithOutput(
          &SocketsInstance::OnTCPServerReceiveCallback);
  pp::CompletionCallbackWithOutput<std::string> on_server_accept_callback =
      callback_factory_.NewCallbackWithOutput(
          &SocketsInstance::OnTCPServerAcceptCallback);

  // If previous server existed, then after this call it will close
  // all existing connections on it.
  tcp_server_.Reset();

  // Start listening on specified port.
  tcp_server_.Listen(port, on_server_accept_callback,
                     on_server_receive_callback);
}
