/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * @file  audio.cc
 *
 * @brief
 * This module demonstrates usage of pp::Audio interface. Besides simple
 * WAVE file loading and playing, it also shows how to mix together more
 * than one sound source.
 * More information about WAVE header can be found on:
 * http://soundfile.sapp.org/doc/WaveFormat/
 */

#include "audio.h"

#include <algorithm>
#include <cmath>
#include <cstring>
#include <limits>
#include <sstream>
#include <utility>

#include "logger.h"

// constant literals
const char* kPlayCommand = "play";
const char* kPauseCommand = "pause";
const char* kStopCommand = "stop";
const char* kLoadCommand = "load";
const char* kFileLoadedMessage = "-file loaded";
const char* kErrorMessage = "-error";
const char* kPlayMessage = "-play";
const char* kPauseMessage = "-pause";
const char* kStopMessage = "-stop";
const char* kReturnStopMessage = "-returnStop";
const char* kSoundsFolder = "resources/sounds/";
const char* kHeaderChunkId = "RIFF";
const char* kHeaderFormat = "WAVE";
const char* kHeaderSubchunk1Id = "fmt ";
const char* kHeaderSubchunk2Id = "data";
const char* kChangeBufferSize = "change";
const char* kVolume = "volume";
const char* kBothChannelMessage = "-channel both";
const char* kLeftChannelMessage = "-channel left";
const char* kRightChannelMessage = "-channel right";
const char* kLatencyMessage = "-latency";
const char* kSampleMessage = "-sample";
const char* kFrameMessage = "-frame";
const char* kChangeSampleRateTo44k = "-rate 44100";
const char* kChangeSampleRateTo48k = "-rate 48000";
const char* kChangeRadioSampleRate = "-change radio";

/** Utility functions */
namespace {

bool StartsWith(const std::string& s, const char* prefix) {
  return (s.find(prefix) == 0);
}

}

SoundInstance::SoundInstance(int id, uint16_t channels,
      uint32_t sample_data_size, uint16_t* sample_data) :
    id_(id),
    channels_(channels),
    sample_data_size_(sample_data_size),
    sample_data_offset_(0),
    is_active_(false) {
  sample_data_.reset(new uint16_t[sample_data_size]);
  memcpy(sample_data_.get(), sample_data,
      sample_data_size_ * sizeof(uint16_t));
}

void SoundInstance::Activate() {
  if (!is_active_) {
    is_active_ = true;
  }
}

void SoundInstance::Deactivate() {
  if (is_active_) {
    is_active_ = false;
  }
}

AudioInstance::AudioInstance(PP_Instance instance) :
    pp::Instance(instance),
    header_(NULL),
    file_number_(0),
    sample_frame_count_(0),
    volume_boost_(DEFAULT_VOLUME),
    callback_factory_(this),
    used_channels_(kBothChannels),
    sample_frame_count_sum_(0),
    latency_sum_(0) {
  file_names_.reserve(kSoundSampleMax);
  Logger::InitializeInstance(this);
}

bool AudioInstance::Init(uint32_t /*argc*/, const char* /*argn*/[],
    const char* /*argv*/[]) {
  // getting sample frame count from the browser
  sample_frame_count_ =
    pp::AudioConfig::RecommendSampleFrameCount(this, PP_AUDIOSAMPLERATE_44100,
                                               SAMPLE_FRAME_COUNT);
  // creating an audio resource with configuration using 44,1kHz sample rate
  // and sample frame count from the browser
  pp::AudioConfig audio_config = pp::AudioConfig(this, PP_AUDIOSAMPLERATE_44100,
                                                 sample_frame_count_);
  audio_ = pp::Audio(this, audio_config, AudioCallback, this);
  return true;
}

void AudioInstance::HandleMessage(const pp::Var& var_message) {
  if (var_message.is_string()) {
    const std::string message = var_message.AsString();
    if (StartsWith(message, kPlayCommand)) {
      // start playing
      Play(static_cast<int>(GetValueFromMessage(message, kPlayCommand)));
    } else if (StartsWith(message, kPauseCommand)) {
      // pause
      Pause(static_cast<int>(GetValueFromMessage(message, kPauseCommand)));
    } else if (StartsWith(message, kStopCommand)) {
      // stop
      Stop(static_cast<int>(GetValueFromMessage(message, kStopCommand)));
    } else if (StartsWith(message, kBothChannelMessage)) {
      // playback on both channels
        used_channels_ = kBothChannels;
        Logger::Log("channels: both ");
    } else if (StartsWith(message, kLeftChannelMessage)) {
      // playback on left channel only
        used_channels_ = kLeftChannel;
        Logger::Log("channels: left ");
    } else if (StartsWith(message, kRightChannelMessage)) {
      // playback on right channel only
        used_channels_ = kRightChannel;
        Logger::Log("channels: right ");
    } else if (StartsWith(message, kChangeSampleRateTo44k)) {
        ChangeAudioSampleRate(PP_AUDIOSAMPLERATE_44100);
    } else if (StartsWith(message, kChangeSampleRateTo48k)) {
        ChangeAudioSampleRate(PP_AUDIOSAMPLERATE_48000);
    } else if (StartsWith(message, kLoadCommand)) {
      // load file
      PrepareReadingFile(message.substr(strlen(kLoadCommand)));
    } else if (StartsWith(message, kChangeBufferSize)) {
      // change buffer size
      ChangeAudioBufferSize(static_cast<int>(
          GetValueFromMessage(message, kChangeBufferSize)));
    } else if (StartsWith(message, kVolume)) {
      volume_boost_ = GetValueFromMessage(message, kVolume);
      Logger::Log("Volume changed to %f", volume_boost_);
    }
  }
}

template <typename T>
std::string AudioInstance::CreateCommandMessage(const char* command,
    T number) {
  std::stringstream message;
  message << command << number;
  return message.str();
}

float AudioInstance::GetValueFromMessage(const std::string& message,
    const char* command) {
  float retVal = -1;
  std::istringstream iss(message.substr(strlen(command)));
  iss >> retVal;
  return retVal;
}

std::string AudioInstance::CheckForWAVEHeaderErrors(
    const WAVEFileHeader& header) {
  std::stringstream error;
  if (strncmp(header.chunk_id, kHeaderChunkId, strlen(kHeaderChunkId)) != 0) {
    error << "\tHeader should start with \"RIFF\" letters.\n";
  }
  if (strncmp(header.format, kHeaderFormat, strlen(kHeaderFormat)) != 0) {
    error << "\tFormat should be \"WAVE\".\n";
  }
  if (strncmp(header.subchunk1_id, kHeaderSubchunk1Id,
      strlen(kHeaderSubchunk1Id)) != 0) {
    error << "\tSubchunk1ID should contain letters \"fmt \".\n";
  }
  if (strncmp(header.subchunk2_id, kHeaderSubchunk2Id,
      strlen(kHeaderSubchunk2Id)) != 0) {
    error << "\tSubchunk2ID should contain letters \"data\".\n";
  }
  if (header.audio_format != 1) {
    error << "\tAudio format should be PCM, "
        << "but it is " << header.audio_format << ".\n";
  }
  if (header.bits_per_sample != 16) {
    error << "\tBits per sample should be 16, but it is "
        << header.bits_per_sample << ".\n";
  }
  if (header.subchunk1_size != 16) {
    error << "\tSize of subchunk1 should be 16, but is "
        << header.subchunk1_size << ".\n";
  }
  if (header.chunk_size != 4 + (8 + header.subchunk1_size)
      + (8 + header.subchunk2_size)) {
    error << "\tChunk size should be " << (8 + header.subchunk1_size) +
        (8 + header.subchunk2_size) << ", but is "<< header.chunk_size
        << ".\n";
  }
  if (header.num_channels != 1 && header.num_channels != 2) {
    error << "\tSupporting only mono and stereo sound, but number of "
        "channels equals " << header.num_channels << ".\n";
  }
  if (header.byte_rate != header.sample_rate * header.num_channels *
      header.bits_per_sample / 8) {
    error << "\tByte rate should be " << header.sample_rate *
        header.num_channels * header.bits_per_sample / 8 << ", but is "
        << header.byte_rate << ".\n";
  }
  if (header.block_align != header.num_channels * header.bits_per_sample / 8) {
    error << "\tBlock align should be " << header.num_channels *
        header.bits_per_sample / 8 << ", but is " << header.block_align
        << ".\n";
  }
  return error.str();
}

void AudioInstance::ReadWAVE(const std::string& data) {
  Logger::Log("Interpreting WAVE data of file %s",
      file_names_[file_number_].c_str());

  // clean and create buffer for interpreting data read from file
  file_data_bytes_.reset(new char[data.size() + 1]);

  // copying data read from file to buffer
  std::copy(data.begin(), data.end(), file_data_bytes_.get());
  file_data_bytes_[data.size()] = '\0';

  // reading header data
  header_ = reinterpret_cast<WAVEFileHeader*>(file_data_bytes_.get());

  // check for malformed header
  std::string header_error = CheckForWAVEHeaderErrors(*header_);
  if (!header_error.empty()) {
    Logger::Error("Unsupported file %s - bad header:\n%s",
        file_names_[file_number_].c_str(), header_error.c_str());

    PostMessage(CreateCommandMessage<int>(kErrorMessage, file_number_));
    return;
  }
  
  uint16_t* sample_data(reinterpret_cast<uint16_t*>(
      file_data_bytes_.get() + sizeof(WAVEFileHeader)));

  // check what sample rate is required by header and set this value if possible
  CheckAndChangeSampleRate(*header_);

  // create new sound instance, fill it's fields and add it
  // to a sound instances vector
  std::shared_ptr<SoundInstance> instance(new SoundInstance(file_number_,
      header_->num_channels, (data.size() - sizeof(WAVEFileHeader)) /
      sizeof(uint16_t), sample_data));
  sound_instances_.insert(std::pair<int, std::shared_ptr<SoundInstance> >(
      instance->id_, instance));

  // depending on current file reading send appropriate control message
  PostMessage(CreateCommandMessage<int>(kFileLoadedMessage, file_number_));

  Logger::Log("File %s data loaded",
      file_names_[file_number_].c_str());
}

void AudioInstance::CheckAndChangeSampleRate(const WAVEFileHeader& header) {
  if (header.sample_rate == audio_.config().sample_rate()) {
      Logger::Log("No need to change sample rate");
      return;
  }
  if (header.sample_rate!= PP_AUDIOSAMPLERATE_44100 &&
          header.sample_rate != PP_AUDIOSAMPLERATE_48000) {
      Logger::Log("Unsupported sample rate. Leaving previous one: %d",
                   audio_.config().sample_rate());
      return;
  }
  if (header.sample_rate == PP_AUDIOSAMPLERATE_44100)
      ChangeAudioSampleRate(PP_AUDIOSAMPLERATE_44100);
  else
      ChangeAudioSampleRate(PP_AUDIOSAMPLERATE_48000);
}

void AudioInstance::ChangeAudioSampleRate(PP_AudioSampleRate sample_rate) {
  // creating new audio config with given sample rate
  pp::AudioConfig new_audio_config = pp::AudioConfig(this, sample_rate,
                                                     sample_frame_count_);
  if (GetActiveSoundsNumber() > 0)
    StopAudioPlayback();

  audio_ = pp::Audio(this, new_audio_config, AudioCallback, this);

  if (GetActiveSoundsNumber() > 0)
    StartAudioPlayback();

  Logger::Log("Changed audio sample rate to: %d",
              audio_.config().sample_rate());
  PostMessage(CreateCommandMessage<PP_AudioSampleRate>(kChangeRadioSampleRate,
              audio_.config().sample_rate()));
}

void AudioInstance::ChangeAudioBufferSize(uint32_t count) {
  uint32_t new_sample_frame_count =
    pp::AudioConfig::RecommendSampleFrameCount(this, PP_AUDIOSAMPLERATE_44100,
                                               count);
  if (new_sample_frame_count != sample_frame_count_) {
    sample_frame_count_ = new_sample_frame_count;
    // creating an audio resource with configuration using 44,1kHz sample rate
    // and sample frame count from the browser closest to the given value
    pp::AudioConfig audio_config = pp::AudioConfig(this,
                                                   PP_AUDIOSAMPLERATE_44100,
                                                   sample_frame_count_);
    if (GetActiveSoundsNumber() > 0)
      StopAudioPlayback();
    audio_ = pp::Audio(this, audio_config, AudioCallback, this);
    if (GetActiveSoundsNumber() > 0)
      StartAudioPlayback();
    Logger::Log("Changed audio buffer size to %d", sample_frame_count_);
  } else {
    Logger::Log("No need to change audio buffer size");
  }
}

void AudioInstance::StartAudioPlayback() {
  // when playback starts, all parameters are set to 0
  sample_frame_count_sum_ = 0;
  latency_sum_ = 0;
  callback_counter = 0;
  PostMessage(CreateCommandMessage<PP_TimeDelta>(kLatencyMessage, 0));
  PostMessage(CreateCommandMessage<uint32_t>(kFrameMessage, 0));
  audio_.StartPlayback();
}

void AudioInstance::StopAudioPlayback() {
  // when playback stops, parameters are displayed
  audio_.StopPlayback();
  PP_TimeDelta latency_ = latency_sum_ / callback_counter;
  PostMessage(CreateCommandMessage<PP_TimeDelta>(kLatencyMessage, latency_));
  uint32_t real_frame_count_ = sample_frame_count_sum_ / callback_counter;
  PostMessage(CreateCommandMessage<uint32_t>(kFrameMessage, real_frame_count_));
}

void AudioInstance::Play(int soundId) {
  assert((soundId > 0) && (soundId <= kSoundSampleMax));
  sound_instances_[soundId]->Activate();
  // start playing if handling first active sound
  if (GetActiveSoundsNumber() == 1) {
    StartAudioPlayback();
    Logger::Log("Playing started");
  }
  // the "boing" sound is clickable many times, we want every time
  // to play from the beginning
  if (soundId == kSoundSampleBoing) {
    sound_instances_[soundId]->sample_data_offset_ = 0;
  }
  // we don't want to send a message for the "boing" sound
  if (soundId != kSoundSampleBoing) {
    // send message to browser, that playing started
    PostMessage(CreateCommandMessage<int>(kPlayMessage, soundId));
  }
}

void AudioInstance::Pause(int soundId) {
  assert((soundId > 0) && (soundId <= kSoundSampleMax));
  // stop playing if handling last active sound
  if (GetActiveSoundsNumber() == 1) {
    StopAudioPlayback();
    Logger::Log("Playing stopped");
  }
  // set clicked sound as not active
  sound_instances_[soundId]->Deactivate();
  // send message to browser, that playing is paused
  PostMessage(CreateCommandMessage<int>(kPauseMessage, soundId));
}

void AudioInstance::Stop(int soundId) {
  assert((soundId > 0) && (soundId <= kSoundSampleMax));
  // stop playing if handling last active sound
  if (GetActiveSoundsNumber() == 1) {
    StopAudioPlayback();
    Logger::Log("Playing stopped");
  }
  // set clicked sound as not active
  sound_instances_[soundId]->Deactivate();
  // set offset to 0 to start playing from the beginning
  // the next time play is requested
  sound_instances_[soundId]->sample_data_offset_ = 0;

  // we don't want to send a message for the last sound
  if (soundId < kSoundSampleBoing) {
    // send message to browser, that playing stopped
    PostMessage(CreateCommandMessage<int>(kStopMessage, soundId));
  }
}

int AudioInstance::GetActiveSoundsNumber() {
  int active_sounds = 0;
  for (SoundInstances::iterator it = sound_instances_.begin();
       it != sound_instances_.end(); it++) {
    if (it->second->is_active_)
      ++active_sounds;
  }
  return active_sounds;
}

void AudioInstance::AudioCallback(void* samples, uint32_t buffer_size,
                                  PP_TimeDelta latency, void* data) {
  // instance pointer is passed when creating Audio resource
  AudioInstance* audio_instance = reinterpret_cast<AudioInstance*>(data);

  // parameters necessary for displaying frame count and latency are set
  ++audio_instance->callback_counter;
  audio_instance->latency_sum_ += latency;
  audio_instance->sample_frame_count_sum_ +=
                                    audio_instance->sample_frame_count_;

  // browser audio buffer to fill
  int16_t* buff = reinterpret_cast<int16_t*>(samples);
  memset(buff, 0, buffer_size);

  assert(audio_instance->GetActiveSoundsNumber() > 0);
  // compute volume of sound instances
  const double volume =
    audio_instance->volume_boost_ /
    sqrt(audio_instance->GetActiveSoundsNumber());
  // Make sure we can't write outside the buffer
  assert(buffer_size >= (sizeof(*buff) * MAX_CHANNELS_NUMBER *
      audio_instance->sample_frame_count_));

  for (SoundInstances::iterator it =
      audio_instance->sound_instances_.begin(); it !=
          audio_instance->sound_instances_.end(); ++it) {
    std::shared_ptr<SoundInstance> instance(it->second);

    // main loop for writing samples to audio buffer
    for (size_t i = 0;  i < audio_instance->sample_frame_count_; ++i) {
      // if there are samples to play
      if (audio_instance->sound_instances_[instance->id_]->is_active_ &&
          instance->sample_data_offset_ < instance->sample_data_size_) {
        if (audio_instance->used_channels_ & kLeftChannel) {
          audio_instance->SafeAdd(&buff[2 * i], volume *
              (int16_t)instance->sample_data_[instance->sample_data_offset_]);
        }
        // for mono sound each sample is passed to both channels, but for
        // stereo samples are written successively to all channels
        if (instance->channels_ == 2)
          ++instance->sample_data_offset_;
        if (audio_instance->used_channels_ & kRightChannel) {
          audio_instance->SafeAdd(&buff[2 * i + 1],  volume *
              (int16_t)instance->sample_data_[instance->sample_data_offset_]);
        }
        ++instance->sample_data_offset_;
      }
    }

    // playing finished so stop it
    if (instance->sample_data_offset_ == instance->sample_data_size_) {
      // Normally we should avoid Pepper API calls in audio callback,
      // as it may result in an audio callback thread being swapped out
      // which leads to audio dropouts. Here the Pepper API is called
      // when audio data ends, so a dropout wouldn't bother us.
      //
      audio_instance->PostMessage(audio_instance->CreateCommandMessage<int>(
          kReturnStopMessage, instance->id_));
    }
  }
}

void AudioInstance::SafeAdd(int16_t *dest, int16_t arg) const {
  int32_t value = *dest;
  value += arg;
  // make sure we don't write to big or too small value
  value = std::min(static_cast<int32_t>(std::numeric_limits<int16_t>::max()),
      static_cast<int32_t>(value));
  value = std::max(static_cast<int32_t>(std::numeric_limits<int16_t>::min()),
      static_cast<int32_t>(value));
  *dest = value;
}

void AudioInstance::PrepareReadingFile(const std::string &file_name_command) {
  // clean old and create new resources for reading from file
  url_request_.reset(new pp::URLRequestInfo(this));
  url_loader_.reset(new pp::URLLoader(this));
  buffer_.reset(new char[READ_BUFFER_SIZE]);
  file_data_ = "";

  // extract file id and name
  std::string file_number = file_name_command.substr(0,
      file_name_command.find(" "));
  std::string file_name = file_name_command.substr(file_number.length() + 1,
      file_name_command.size() - file_number.size() - 1);
  file_number_ = atoi(file_number.c_str());
  if (file_names_.size() <= file_number_)
    file_names_.resize(file_number_ + 1);
  file_names_[file_number_] = file_name;

  // set file to download
  url_ = kSoundsFolder + file_names_[file_number_];
  url_request_->SetURL(url_);
  url_request_->SetMethod("GET");
  url_request_->SetRecordDownloadProgress(true);

  // start reading from file
  pp::CompletionCallback open_callback =
      callback_factory_.NewCallback(&AudioInstance::OnOpenURLCallback);
  url_loader_->Open(*url_request_, open_callback);
  Logger::Log("Reading file " + file_name);
}

void AudioInstance::OnOpenURLCallback(int32_t result) {
  // if file haven't opened correctly
  if (result != PP_OK) {
    FileReadCompleted(false);
    return;
  }
  Logger::Log("Opened file %s", file_names_[file_number_].c_str());
  // start reading file data
  ReadFile();
}

void AudioInstance::ReadFile(void) {
  pp::CompletionCallback callback =
      callback_factory_.NewOptionalCallback(&AudioInstance::OnReadFileCallback);
  int32_t result = PP_OK;

  // read parts of file until whole file is read
  while (true) {
    result = url_loader_->ReadResponseBody(buffer_.get(), READ_BUFFER_SIZE,
        callback);
    if (result > 0) {
      AppendData(result);
    } else {
      break;
    }
  }

  // if something went wrong
  if (result != PP_OK_COMPLETIONPENDING) {
    callback.Run(result);
  }
}
void AudioInstance::OnReadFileCallback(int32_t result) {
  // whole file was read
  if (result == PP_OK) {
    FileReadCompleted(true);
  } else if (result > 0) {  // still there is something to read
    AppendData(result);
    ReadFile();
  } else {  // an error occurred
    FileReadCompleted(false);
  }
}

void AudioInstance::AppendData(int32_t bytes_count) {
  // save data read from file
  file_data_.insert(file_data_.end(), buffer_.get(),
      buffer_.get() + bytes_count);
}

void AudioInstance::FileReadCompleted(bool result) {
  // clear the buffer
  buffer_.reset();

  // an error occurred
  if (!result) {
    Logger::Error("Could not read file %s",
        file_names_[file_number_].c_str());

    PostMessage(CreateCommandMessage<int>(kErrorMessage, file_number_));
  } else {  // file was read correctly
    Logger::Log("File %s reading complete",
        file_names_[file_number_].c_str());
    // interpret data of audio file
    ReadWAVE(file_data_);
  }
}

class AudioModule: public pp::Module {
 public:
  AudioModule() : pp::Module() {
  }
  virtual ~AudioModule() {
  }

  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new AudioInstance(instance);
  }
};

namespace pp {

Module* CreateModule() {
  return new AudioModule();
}

}  // namespace pp
