/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef GRAPHICS_2D_SRC_GRAPHICS_2D_H_
#define GRAPHICS_2D_SRC_GRAPHICS_2D_H_

#include "ppapi/c/ppb_image_data.h"
#include "ppapi/cpp/graphics_2d.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/point.h"
#include "ppapi/cpp/size.h"
#include "ppapi/utility/completion_callback_factory.h"

/**
 * This is a NaCl application that shows a 2D flame animation with use of
 * Graphics2D PPAPI.
 */
class Graphics2DFlameInstance : public pp::Instance {
 public:
  /**
   * All public methods below are part of pp::Instance class interface.
   * For more info @see Instance .
   */
  explicit Graphics2DFlameInstance(PP_Instance instance);
  ~Graphics2DFlameInstance();
  virtual bool Init(uint32_t argc, const char* argn[], const char* argv[]);
  virtual void DidChangeView(const pp::View& view);
  virtual bool HandleInputEvent(const pp::InputEvent& event);

 private:
  /**
   * A complete iteration done for every animation frame.
   * Updates the image to be displayed and flushes it onto the screen.
   * @param int32_t - is not used, but has to be declared to fit a
   * CompletionCallback function pattern.
   */
  void MainLoopIteration(int32_t);

  /**
   * Initializes a palette of colors specific for our flame animation.
   */
  void CreatePalette();

  /**
   * Creates a Graphics2D context with given size and binds it to this
   * Graphics2DFlameInstance object.
   * @param new_size  Graphics view size.
   * @return true if Graphics2D was created successfully.
   */
  bool CreateContext(const pp::Size& new_size);

  /**
   * Updates the buffer_ with next frame of flame animation.
   */
  void Update();

  /**
   * Generates random flames at the bottom of the graphics view.
   */
  void UpdateCoals();

  /**
   * Updates flames, so they float to the top of view.
   */
  void UpdateFlames();

  /**
   * Draws a flame around the mouse cursor if left mouse button is pressed.
   */
  void DrawMouse();

  /**
   * Translates the image from buffer_ into a raw image data in proper format.
   * Flushes the next frame into the screen.
   */
  void Paint();

  pp::CompletionCallbackFactory<Graphics2DFlameInstance> callback_factory_;
  /**< Simplifies a CompletionCallback creation. */
  pp::Graphics2D context_; /**< Graphics2D context. */
  pp::Graphics2D flush_context_; /**< Auxiliary copy of Graphics2D context. */
  pp::Size size_; /**< Graphics view size. */
  pp::Point mouse_; /**< Mouse position relative to left top view corner. */
  bool mouse_down_; /**< Indicates if mouse button is pressed down currently. */
  uint8_t* buffer_;
  /**< A buffer of palette entries of the same size as context_.size() . */
  uint32_t palette_[256]; /**< Palette of colors used to draw images. */
  float device_scale_; /**< @see View.GetDeviceScale() */
};

class Graphics2DFlameModule : public pp::Module {
 public:
  Graphics2DFlameModule()
      : pp::Module() {
  }

  virtual ~Graphics2DFlameModule() {
  }

  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new Graphics2DFlameInstance(instance);
  }
};

namespace pp {
Module* CreateModule() {
  return new Graphics2DFlameModule();
}
}  // namespace pp

#endif /* GRAPHICS_2D_SRC_GRAPHICS_2D_H_ */
