/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SOCKETS_SRC_SIMPLE_TCP_SERVER_H_
#define SOCKETS_SRC_SIMPLE_TCP_SERVER_H_

#include <string>

#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/tcp_socket.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "config.h"

/**
 * Simple server based on a listening pp::TCPSocket.
 * This server handles just one connection at a time and will
 * run on_server_accept_callback_ on accepting an incoming
 * connection or on_server_receive_callback_ when receiving
 * a message on the listening socket.
 */
class SimpleTCPServer {
 public:
  SimpleTCPServer(pp::Instance* instance)
      : instance_handle_(instance),
        callback_factory_(this),
        on_server_accept_callback_(
            pp::CompletionCallbackWithOutput<std::string>(0)),
        on_server_receive_callback_(
            pp::CompletionCallbackWithOutput<std::string>(0)) {
  }

  /**
   * Resets active sockets. Calling this method will cause
   * the server to stop listening and close active connections.
   */
  void Reset();

  /**
   * Checks if PPAPI interfaces required for
   * this class are available.
   *
   * @return True if required interfaces are available.
   */
  static bool InterfacesAreAvailable();

  /**
   * Starts listening on a specific port.
   *
   * @param [in] port Port to listen on.
   * @param [in] on_server_accept_callback A callback called when server accepts a connection.
   * @param [in] on_server_receive_callback A callback called when server receives
   *             a message on an active connection.
   */
  void Listen(uint16_t port,
      pp::CompletionCallbackWithOutput<std::string> on_server_accept_callback,
      pp::CompletionCallbackWithOutput<std::string> on_server_receive_callback);

  /**
   * Sends a message on an active socket connection.
   *
   * @param [in] message to be send on the socket.
   * @param [in] on_server_receive_callback is called when new message is received.
   */
  void Write(const std::string& message,
      pp::CompletionCallbackWithOutput<std::string> on_server_receive_callback);

 private:
  /**
   * Invokes reading on a connected socket.
   */
  void TryRead();

  /**
   * Invokes accept on a listeing server
   */
  void TryAccept();

  /**
   * Callback functions
   */
  void OnBindCompletion(int32_t result);
  void OnListenCompletion(int32_t result);
  void OnAcceptCompletion(int32_t result, pp::TCPSocket socket);
  void OnReadCompletion(int32_t result);
  void OnWriteCompletion(int32_t result);

  pp::InstanceHandle instance_handle_;

  /**
   * Callback factory class used for easier and more convenient
   * callback creations.
   */
  pp::CompletionCallbackFactory<SimpleTCPServer> callback_factory_;

  /**
   * Holds callback for calling upon connection acceptation
   */
  pp::CompletionCallbackWithOutput<std::string> on_server_accept_callback_;

  /**
   * Holds callback for calling when receiving a message from
   * a connected socket.
   */
  pp::CompletionCallbackWithOutput<std::string> on_server_receive_callback_;

  /**
   * Socket used as a server.
   */
  pp::TCPSocket listening_socket_;

  /**
   * Socket used for saving incoming connections to server.
   */
  pp::TCPSocket incoming_socket_;

  /**
   * Buffer used for receiving a message from a socket.
   */
  char receive_buffer_[kBufferSize];
};

#endif  // SOCKETS_SRC_SIMPLE_TCP_SERVER_H_
