/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SOCKETS_SRC_SIMPLE_UDP_CLIENT_H_
#define SOCKETS_SRC_SIMPLE_UDP_CLIENT_H_

#include <string>

#include "ppapi/cpp/host_resolver.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/udp_socket.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "config.h"

#define UNUSED(x) (void)(x)

/**
 * Simple UDP client based on pp::UDPSocket.
 * This client after connecting will go straight into
 * the receiving mode. After receiving a message on socket,
 * it will invoke a user custom callback on_receive_from_callback_.
 * It allows also to send a message to the destination host passed
 * as a parameter in Bind method.
 */
class SimpleUDPClient {
 public:
  SimpleUDPClient(pp::Instance* instance)
    : instance_handle_(instance),
      callback_factory_(this),
      on_receive_from_callback_(NULL),
      send_outstanding_(false) {
  }

  /**
   * Checks if PPAPI interfaces required for
   * this class are available.
   *
   * @return True if required interfaces are available.
   */
  static bool InterfacesAreAvailable();

  /**
   * Binds to "Any" local address and port (This is
   * equivalent behaviour to posix flag INADDR_ANY).
   *
   * @param [in] destination_host Remote destination host address.
   * @param [in] receive_from_callback A callback called when new message is received.
   */
  void Bind(const std::string& destination_host,
      pp::CompletionCallbackWithOutput<std::string> receive_from_callback);

  /**
   * Sends the message through socket.
   *
   * @param [in] message Message to be send.
   */
  void SendTo(const std::string& message);

  /**
   * Destroys socket and cancels all pending
   * SendTo and Receive callbacks.
   */
  void Close();

  /**
   * Receives a message from a socket.
   *
   * @param [in] callback on_receive_from_callback A callback called when a message
   *             was received on socket.
   */
  void Receive(
      pp::CompletionCallbackWithOutput<std::string> on_receive_from_callback);

 private:
  bool IsBound();

  /**
   * Callbacks functions
   */
  void OnResolveCompletion(int32_t result);
  void OnBindCompletion(int32_t result);
  void OnReceiveFromCompletion(int32_t result, pp::NetAddress source);
  void OnSendToCompletion(int32_t result);

  pp::InstanceHandle instance_handle_;

  /**
   * Callback factory class used for easier and more convenient
   * callback creations.
   */
  pp::CompletionCallbackFactory<SimpleUDPClient> callback_factory_;

  /**
   * This callback is called after we receive a message on socket.
   */
  pp::CompletionCallbackWithOutput<std::string> on_receive_from_callback_;

  /**
   * Resolver is used to resolve host name to IP.
   */
  pp::HostResolver resolver_;

  /**
   * Object that provide UDP Socket functionality.
   */
  pp::UDPSocket udp_socket_;

  /**
   * Address of the remote host to which we will send messages.
   */
  pp::NetAddress remote_host_;

  /**
   * Buffer which is passed when receiving a message.
   */
  char receive_buffer_[kBufferSize];
  bool send_outstanding_;
};

#endif  // SOCKETS_SRC_SIMPLE_UDP_CLIENT_H_
