/**
 * Copyright (c) 2013 The Chromium Authors. All rights reserved.
 * Copyright (c) 2015, Samsung Electronics Co., Ltd. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Google Inc, Samsung Electronics nor the names of
 *    its contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SOCKETS_SRC_SOCKETS_H_
#define SOCKETS_SRC_SOCKETS_H_

#include <stdio.h>
#include <sstream>
#include <string>

#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"

#include "simple_tcp_client.h"
#include "simple_tcp_server.h"
#include "simple_udp_client.h"
#include "simple_udp_server.h"

/**
 * This message might either specify new TCP connection
 * or define the TCP type of socket for other commands.
 */
#define MSG_TCP 't'

/**
 * This message might either specify new UDP connection
 * or define the UDP type of socket for other commands.
 */
#define MSG_UDP 'u'

/**
 * Send a message on socket
 */
#define MSG_SEND 's'

/**
 * Close socket
 */
#define MSG_CLOSE 'c'

/**
 * Listen on socket
 */
#define MSG_LISTEN 'l'

class SocketsInstance : public pp::Instance {
 public:
  explicit SocketsInstance(PP_Instance instance)
      : pp::Instance(instance),
        current_tcp_port_(-1),
        callback_factory_(this),
        tcp_client_(this),
        tcp_server_(this),
        udp_client_(this),
        udp_server_(this) {
  }

  /**
   * This method is called upon plugin initialization.
   * It`s checking whether it is possible to access
   * required interfaces for this plugin.
   */
  virtual bool Init(uint32_t argc, const char* argn[], const char* argv[]);

  /**
   * Method that handles incoming messages from JavaScript side.
   */
  virtual void HandleMessage(const pp::Var& var_message);

  /**
   * TCP Callback functions
   */
  void OnTCPClientReceiveCallback(int32_t result, const std::string& message);
  void OnTCPServerReceiveCallback(int32_t result, const std::string& message);
  void OnTCPServerAcceptCallback(int32_t result, const std::string& ip);

  /**
   * UDP Callback functions
   */
  void OnUDPClientReceiveFromCallback(int32_t result, const std::string& message);
  void OnUDPServerReceiveCallback(int32_t result,
                                  struct MessageInfo extended_message);

  virtual ~SocketsInstance() { }

 private:

  /**
   * Starts listening on specific port.
   */
  void StartListenOnTCPSocket(int port);

  /**
   * Holds information about currently used port by TCP Socket.
   */
  int current_tcp_port_;

  /**
   * Callback factory class used for easier and more convenient
   * callback creation.
   */
  pp::CompletionCallbackFactory<SocketsInstance> callback_factory_;

  /**
   * Represents TCP client used for connecting, sending and
   * receiving messages.
   */
  SimpleTCPClient tcp_client_;

  /**
   * TCP server used for handling incoming connections and
   * sending/receiving messages from them.
   */
  SimpleTCPServer tcp_server_;

  /**
   * Represents UDP client used sending and receiving
   * messages.
   */
  SimpleUDPClient udp_client_;

  /**
   * UDP server used for binding to local address and then
   * receiving/sending messages from/to remote hosts.
   */
  SimpleUDPServer udp_server_;
};

/**
 * The ExampleModule provides an implementation of pp::Module that creates
 * ExampleInstance objects when invoked.
 */
class SocketsModule : public pp::Module {
 public:
  SocketsModule() : pp::Module() {}
  virtual ~SocketsModule() {}

  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new SocketsInstance(instance);
  }
};

/**
 * Implement the required pp::CreateModule function that creates our specific
 * kind of Module.
 */
namespace pp {
Module* CreateModule() { return new SocketsModule(); }
}  // namespace pp

#endif  // SOCKETS_SRC_SOCKETS_H_
