/**
 * Copyright (c) 2015, Samsung Electronics Co., Ltd
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Samsung Electronics nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "own_thread_url_loader.h"

#include <sstream>
#include <string>

#include "ppapi/cpp/completion_callback.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/instance_handle.h"
#include "ppapi/cpp/message_loop.h"
#include "ppapi/cpp/url_loader.h"
#include "ppapi/cpp/url_request_info.h"
#include "ppapi/cpp/url_response_info.h"
#include "ppapi/cpp/var.h"
#include "ppapi/utility/completion_callback_factory.h"
#include "ppapi/utility/threading/simple_thread.h"

namespace {
const int32_t kHttpOk = 200; // 200 is HTTP status for OK
}

OwnThreadUrlLoader::OwnThreadUrlLoader(const pp::InstanceHandle& instance)
    : instance_(instance)
    , buffer_size_(32768)
    , buffer_(new uint8_t[buffer_size_])
    , cc_factory_(this)
    , loader_(instance)
    , simple_thread_url_(instance) {
}

OwnThreadUrlLoader::~OwnThreadUrlLoader() {
  loader_.Close();
}

void OwnThreadUrlLoader::DownloadInLoaderThread(
    const std::string& url, const pp::CompletionCallbackWithOutput<std::string>& callback) {

  // The message loop from which this function is being stored in order to call
  // from it the callback when the work is done.
  // See SendCallback function.
  caller_loop_ = pp::MessageLoop::GetCurrent();
  caller_callback_.reset(new pp::CompletionCallbackWithOutput<std::string>(callback));
  pp::CompletionCallback cc = cc_factory_.NewCallback(
      &OwnThreadUrlLoader::StartLoading, url);

  simple_thread_url_.Start();
  if(PP_OK != simple_thread_url_.message_loop().PostWork(cc))
    caller_callback_->Run(PP_ERROR_FAILED);
}

void OwnThreadUrlLoader::Download(const std::string& url, const pp::CompletionCallbackWithOutput<std::string>& callback) {
  caller_callback_.reset(new pp::CompletionCallbackWithOutput<std::string>(callback));
  StartLoading(PP_OK, url);
}

void OwnThreadUrlLoader::StartLoading(int32_t result, const std::string& url) {
  pp::URLRequestInfo request(instance_);
  request.SetURL(url);
  request.SetMethod("GET");

  if(caller_loop_.is_null()) {
    // This block will be executed if the Download function was called not by
    // calling the DownloadInLoaderThread function, so it is in its caller thread.
    // The CompletionCallbackFactory::NewCallback() function creates
    // an asynchronous CompletionCallback. Asynchronous calls to PPAPI can be
    // done from threads that have a message loop attached and running.
    loader_.Open(request, cc_factory_.NewCallback(
        &OwnThreadUrlLoader::OpenCallback));

  } else {
    // This block will be executed if the Download function was called from the
    // DownloadInLoaderThread() function. In the side thread we can make PPAPI
    // calls with blocking callbacks. The call will be executed synchronously
    // blocking the thread until it's completed. Such calls can't be done from
    // the main PPAPI thread. The blocking completion callback is created
    // by calling the constructor without parameters.
    loader_.Open(request, cc_factory_.NewCallback(
        &OwnThreadUrlLoader::OpenCallback));
  }
}

void OwnThreadUrlLoader::OpenCallback(int32_t result) {
  if (result != PP_OK) {
    SendCallback(result);
    return;
  }
  pp::URLResponseInfo response_info = loader_.GetResponseInfo();
  if (response_info.is_null() || response_info.GetStatusCode() != kHttpOk) {
    SendCallback(PP_ERROR_FAILED);
    return;
  }
  ReadData();
}

void OwnThreadUrlLoader::ReadResponseBodyCallback(int32_t result) {
  if (result <= 0) {
    SendCallback(result);
    return;
  }
  AddData(std::string(reinterpret_cast<const char*>(buffer_.get()), result));
  ReadData();
}

void OwnThreadUrlLoader::ReadData() {
  // The optional callback function can work both asynchronously
  // and synchronously. If the function that will call this callback is ready
  // (in this case the data is already downloaded and buffer can be quickly
  // filled) then it returns synchronously and the callback is not called.
  // Otherwise it is performed as a normal callback. See do {} while below.
  pp::CompletionCallback callback = cc_factory_.NewCallback(
      &OwnThreadUrlLoader::ReadResponseBodyCallback);

  int32_t result = PP_OK;
  do {
    result = loader_.ReadResponseBody(
        static_cast<void*>(buffer_.get()), buffer_size_, callback);
    // If the result > 0, the ReadResponseBody function returned
    // synchronously and the callback wasn't called and we have to handle the
    // data here.
    // If the result < -1, we need to handle an error.
    // If the result == -1 (PP_OK_COMPLETION_PENDING), then we have to wait
    // for a callback to run. This callback will be called on the thread from
    // where the call was made, so in this loader thread.
    if (result <= 0)
      break;
    AddData(std::string(reinterpret_cast<const char*>(buffer_.get()), result));
  } while (true);
  if (result != PP_OK_COMPLETIONPENDING) {
    // If the callback was registered it must be called, so if the call with
    // the optional callback ended synchronously we must run this callback manually.
    callback.Run(result);
  }
}

void OwnThreadUrlLoader::SendCallback(int32_t result) {
  // If loop was stored it means we are in loader own thread, so we returns
  // data by callback to caller.
  if (PP_OK != result)
    AddData("Error in URLLoader.\n");
  if(!caller_loop_.is_null()) {
    caller_loop_.PostWork(*caller_callback_);
  } else {
    caller_callback_->Run(PP_OK);
  }
}

void OwnThreadUrlLoader::AddData(const std::string& str) {
  caller_callback_->output()->append(str);
}
