#include <tizen.h>
#include "$(appName).h"
#include <cairo.h>
#include <math.h>

typedef struct appdata {

	Evas_Coord width;
	Evas_Coord height;

	Evas_Object *win;
	Evas_Object *img;

	cairo_surface_t *surface;
	cairo_t *cairo;

} appdata_s;

void cairo_drawing(void *data)
{
	appdata_s *ad = data;

	int d = 0;

	if(ad->width < ad->height)
		d = ad->width;
	else
		d = ad->height;

	/* clear background as white */
	cairo_set_source_rgba(ad->cairo, 1, 1, 1, 1);
	cairo_paint(ad->cairo);

	cairo_translate(ad->cairo, 0.1 * d, 0.1 * d);
	cairo_set_line_width(ad->cairo,2);
	cairo_set_source_rgba(ad->cairo, 0.0, 0.0, 1.0, 1.0);

	cairo_move_to (ad->cairo, 0.2 * d , 0.2 * d);
	cairo_line_to (ad->cairo, 0.4 * d, 0.3 * d);
	cairo_rel_line_to (ad->cairo, 0.2 * d, -0.1 * d);
	cairo_arc (ad->cairo, 0.4 * d, 0.4 * d, 0.2* d * sqrt(2), -0.25 * M_PI, 0.25 * M_PI);
	cairo_rel_curve_to (ad->cairo, -0.2* d, -0.1 * d, -0.2* d, 0.1 * d, -0.4 * d, 0);
	cairo_close_path (ad->cairo);
	cairo_fill(ad->cairo);

	cairo_rectangle(ad->cairo, 0, 0, 0.8 * d, 0.8 * d);
	cairo_stroke(ad->cairo);
	cairo_surface_flush(ad->surface);

	/* display cairo drawin on screen */
	unsigned char * imageData = cairo_image_surface_get_data(cairo_get_target(ad->cairo));
	evas_object_image_data_set(ad->img, imageData);
	evas_object_image_data_update_add(ad->img, 0, 0, ad->width, ad->height);
}

static void
win_delete_request_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	cairo_surface_destroy(ad->surface);
	cairo_destroy(ad->cairo);
	ui_app_exit();
}

static void
win_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static void
win_resize_cb(void *data, Evas *e , Evas_Object *obj , void *event_info)
{
	appdata_s *ad = data;

	if(ad->surface)
	{
		cairo_surface_destroy(ad->surface);
		cairo_destroy(ad->cairo);
		ad->surface = NULL;
		ad->cairo = NULL;
	}

	evas_object_geometry_get(obj, NULL, NULL, &ad->width, &ad->height);
	evas_object_image_size_set(ad->img, ad->width, ad->height);
	evas_object_resize(ad->img, ad->width, ad->height);
	evas_object_show(ad->img);

	if(!ad->surface)
	{
		ad->surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, ad->width, ad->height);
		ad->cairo = cairo_create (ad->surface);
		cairo_drawing(ad);
	}
}

static void
cairo_basic_drawing(appdata_s *ad)
{
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_autodel_set(ad->win, EINA_TRUE);
	if (elm_win_wm_rotation_supported_get(ad->win))
	{
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}
	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);
	eext_object_event_callback_add(ad->win, EEXT_CALLBACK_BACK, win_back_cb, ad);
	evas_object_event_callback_add(ad->win, EVAS_CALLBACK_RESIZE, win_resize_cb, ad);
	evas_object_show(ad->win);

	/* Image */
	ad->img = evas_object_image_filled_add(evas_object_evas_get(ad->win));
	evas_object_show(ad->img);

	evas_object_geometry_get(ad->win, NULL, NULL, &ad->width, &ad->height);
	ad->surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, ad->width, ad->height);
	ad->cairo = cairo_create (ad->surface);
	cairo_drawing(ad);
}

static bool
app_create(void *data)
{
	appdata_s *ad = data;
	cairo_basic_drawing(ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad = {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "app_main() is failed. err = %d", ret);
	}

	return ret;
}
