/*
 * Copyright 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "model/calc.h"

#include "display.h"

#include "utils/resources.h"
#include "utils/logger.h"

#include <stdlib.h>
#include <locale.h>
#include <math.h>

/* Used for non-strict Zero checking */
static const double CALC_ZERO_EPS = 1.0e-24;

enum
{
    /* Use this character if system locale function fails */
    CALC_DEFAULT_DEC_POINT_CHAR = '.',
    CALC_DEC_POINT_STR_SIZE = 2 /* 1 for char and 1 for zero terminator */
};

/* Internal data structure of the calc object */
struct _calc
{
    /* Error flag to indicate critical failure and block computations until reset */
    bool error;

    /* Major state variables of the calculator */
    bool dirty; /* we in the middle of expression calculation */
    bool num_is_new; /* indicates that number in num register is new (entered by user of by unary function) */
    bool in_edit_mode; /* indicates that user is entering number to the display */
    key_type op; /* current operation to apply to operands */

    /* Calculator registers used in the computations */
    double num; /* corresponds to the number on display (not actual in edit mode) */
    double acc; /* left operand of the operation, stores the result of the operation */
    double rep; /* special repeat register, used to repeat last operation on result key */

    /* Display object to handle I/O */
    display disp;

    /* 1 char string to hold current decimal point character */
    char dec_point_str[CALC_DEC_POINT_STR_SIZE];
};

/* Initialises allocated calc instance */
static void _calc_init(calc *obj);
/* Resets calculator to the initial state */
static void _calc_reset(calc *obj);

/* Functions for handling specific key types */
static result_type _calc_handle_etc_key(calc *obj, key_type key);
static result_type _calc_handle_num_key(calc *obj, key_type key);
static result_type _calc_handle_op_key(calc *obj, key_type key);
static result_type _calc_handle_result_key(calc *obj);
static result_type _calc_handle_sign_key(calc *obj);
static result_type _calc_handle_erase_key(calc *obj);

/* Sets calc's error flag member to true if passed error code is critical */
/* Sets error message to the display */
static void _calc_handle_error(calc *obj, result_type error);
/* Ends edit mode (if in edit mode) and stores number from display to num register */
static void _calc_end_edit_mode(calc *obj);
/* Performs current operation on acc as left operand and rv as right operand */
/* Result stored in acc */
static result_type _calc_perform_op(calc *obj, double rv);
/* Performs negate function on the num. Do not use in edit mode */
static void _calc_perform_negate_func(calc *obj);

calc *calc_create(result_type *result)
{
    RETVM_IF(!result, NULL, "result is NULL");

    calc *obj = calloc(1, sizeof(calc));
    if (!obj)
    {
        ERR("Failed to allocate calc_model instance");
        *result = RES_OUT_OF_MEMORY;
        return NULL;
    }

    _calc_init(obj);

    *result = RES_OK;
    return obj;
}

void calc_destroy(calc *obj)
{
    free(obj);
}

void _calc_init(calc *obj)
{
    _calc_reset(obj);
    calc_update_region_fmt(obj);
}

void _calc_reset(calc *obj)
{
    obj->error = false;
    obj->dirty = false;
    obj->num_is_new = false;
    obj->in_edit_mode = false;
    obj->op = CALC_KEY_ADD;

    obj->num = 0.0;
    obj->acc = 0.0;
    obj->rep = 0.0;

    display_reset(&obj->disp);
}

void calc_set_display_change_cb(calc *obj, notify_cb cb, void *cb_data)
{
    RETM_IF(!obj, "obj is NULL");

    obj->disp.change_cb = cb;
    obj->disp.change_cb_data = cb_data;
}

result_type calc_handle_key_press(calc *obj, key_type key)
{
    RETVM_IF(!obj, RES_INTERNAL_ERROR, "obj is NULL");

    if (CALC_KEY_RESET == key)
    {
        _calc_reset(obj);
        return RES_OK;
    }

    RETVM_IF(obj->error, RES_ILLEGAL_STATE, "Calculator is in error state");

    result_type result = RES_INTERNAL_ERROR;

    switch (key & CALC_KEY_TYPE_MASK)
    {
    case CALC_KEY_TYPE_NUM:
        result = _calc_handle_num_key(obj, key);
        break;
    case CALC_KEY_TYPE_OP:
        result = _calc_handle_op_key(obj, key);
        break;
    default:
        result = _calc_handle_etc_key(obj, key);
        break;
    }

    if (RES_OK != result)
    {
        _calc_handle_error(obj, result);
        return result;
    }

    return RES_OK;
}

result_type _calc_handle_etc_key(calc *obj, key_type key)
{
    switch (key)
    {
    case CALC_KEY_RESULT:
        return _calc_handle_result_key(obj);
    case CALC_KEY_SIGN:
        return _calc_handle_sign_key(obj);
    case CALC_KEY_ERASE:
        return _calc_handle_erase_key(obj);
    default:
        return RES_ILLEGAL_ARGUMENT;
    }
}

result_type _calc_handle_num_key(calc *obj, key_type key)
{
    bool reset = false;
    if (!obj->in_edit_mode)
    {
        obj->in_edit_mode = true;
        reset = true;
    }
    return display_enter_key(&obj->disp, key, reset);
}

result_type _calc_handle_op_key(calc *obj, key_type key)
{
    switch (key)
    {
    case CALC_KEY_ADD:
    case CALC_KEY_SUB:
    case CALC_KEY_MUL:
    case CALC_KEY_DIV:
        break;
    default:
        return RES_ILLEGAL_ARGUMENT;
    }

    _calc_end_edit_mode(obj);

    result_type result = RES_OK;

    if (!obj->dirty)
    {
        obj->dirty = true;
        obj->acc = obj->num;
    }
    else if (obj->num_is_new)
    {
        result = _calc_perform_op(obj, obj->num);
    }

    obj->num_is_new = false;
    obj->op = key;

    return result;
}

result_type _calc_handle_result_key(calc *obj)
{
    _calc_end_edit_mode(obj);

    if (obj->dirty)
    {
        obj->dirty = false;
        obj->rep = obj->num;
    }
    else if (obj->num_is_new)
    {
        obj->acc = obj->num;
    }

    obj->num_is_new = false;

    return _calc_perform_op(obj, obj->rep);
}

result_type _calc_handle_sign_key(calc *obj)
{
    if (obj->in_edit_mode)
    {
        display_negate(&obj->disp);
    }
    else
    {
        _calc_perform_negate_func(obj);
    }
    return RES_OK;
}

result_type _calc_handle_erase_key(calc *obj)
{
    RETVM_IF(!obj->in_edit_mode, RES_ILLEGAL_STATE, "Not in edit mode");

    return display_erase(&obj->disp);
}

void _calc_handle_error(calc *obj, result_type error)
{
    obj->error = true;
    switch (error)
    {
    case RES_NUMER_IS_TOO_LONG:
        display_set_str(&obj->disp, STR_DISP_NUMBER_IS_TOO_LONG);
        break;
    case RES_INTERNAL_ERROR:
        display_set_str(&obj->disp, STR_DISP_INTERNAL_ERROR);
        break;
    case RES_DIVISION_BY_ZERO:
        display_set_str(&obj->disp, STR_DISP_DIVISION_BY_ZERO);
        break;
    case RES_UNDEFINED_VALUE:
        display_set_str(&obj->disp, STR_DISP_UNDEFINED_VALUE);
        break;
    default:
        obj->error = false;
        break;
    }
}

void _calc_end_edit_mode(calc *obj)
{
    if (obj->in_edit_mode)
    {
        obj->in_edit_mode = false;
        obj->num = display_get_number(&obj->disp);
        obj->num_is_new = true;
        /* This will remove extra chars from current display (for example: 2.300 -> 2.3; -0.0 -> 0) */
        display_set_number(&obj->disp, obj->num);
    }
}

result_type _calc_perform_op(calc *obj, double rv)
{
    switch (obj->op)
    {
    case CALC_KEY_ADD:
        obj->acc += rv;
        break;
    case CALC_KEY_SUB:
        obj->acc -= rv;
        break;
    case CALC_KEY_MUL:
        obj->acc *= rv;
        break;
    case CALC_KEY_DIV:
        if (fabs(rv) <= CALC_ZERO_EPS)
        {
            return ((fabs(obj->acc) <= CALC_ZERO_EPS) ? RES_UNDEFINED_VALUE : RES_DIVISION_BY_ZERO);
        }
        obj->acc /= rv;
        break;
    default:
        return RES_INTERNAL_ERROR;
    }

    result_type result = display_set_number(&obj->disp, obj->acc);
    RETVM_IF(RES_OK != result, result, "Failed to set display number: f", obj->acc);

    /* Get number back from display to get rounded value */
    obj->num = display_get_number(&obj->disp);
    obj->acc = obj->num;

    return RES_OK;
}

void _calc_perform_negate_func(calc *obj)
{
    if (0.0 != obj->num)
    {
        obj->num = -obj->num;
        (void)display_set_number(&obj->disp, obj->num);
    }
    obj->num_is_new = true;
}

const char *calc_get_display_str(calc *obj)
{
    RETVM_IF(!obj, NULL, "obj is NULL");

    return obj->disp.str;
}

void calc_update_region_fmt(calc *obj)
{
    RETM_IF(!obj, "obj is NULL");

    struct lconv *lc = localeconv();

    char new_dec_point_char = (lc && lc->decimal_point && ('\0' != lc->decimal_point[0])) ?
            lc->decimal_point[0] : CALC_DEFAULT_DEC_POINT_CHAR;

    display_set_dec_point_char(&obj->disp, new_dec_point_char);
    obj->dec_point_str[0] = new_dec_point_char;
}

const char *calc_get_dec_point_str(calc *obj)
{
    RETVM_IF(!obj, NULL, "obj is NULL");

    return obj->dec_point_str;
}
