/*
 * Copyright 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __CALC_H__
#define __CALC_H__

#include "utils/types.h"

/**
 * @brief calc structure to be used in this API
 */
typedef struct _calc calc;

/**
 * @brief Creates and initialises a calc object instance
 * @param[out] result   Pointer to the output variable to store the result of the initialisation
 * @return Not NULL - Initialisation OK (*result == RES_OK)
 *         NULL - Initialisation failed (see *result for the reason of the failure)
 */
calc *calc_create(result_type *result);

/**
 * @brief Destroys and frees the calc instance referenced
 * @param[in] obj   calc structure pointer (if NULL function has no effect)
 */
void calc_destroy(calc *obj);

/**
 * @brief Registers the call-back to be invoked on display string change
 * @param[in] obj       calc structure pointer
 * @param[in] cb        Call-back function pointer (NULL to unregister call-back)
 * @param[in] cb_data   Data pointer to pass to the call-back function
 */
void calc_set_display_change_cb(calc *obj, notify_cb cb, void *cb_data);

/**
 * @brief Gets current display string message
 * @param[in] obj   calc structure pointer
 * @return Pointer to the display string
 */
const char *calc_get_display_str(calc *obj);

/**
 * @brief Handles calculator key press
 * @param[in] obj   calc structure pointer
 * @param[in] key   Key value to handle
 * @return Result code of the operation
 */
result_type calc_handle_key_press(calc *obj, key_type key);

/**
 * @brief Updates region format dependent properties (decimal point character)
 * @param[in] obj   calc structure pointer
 */
void calc_update_region_fmt(calc *obj);

/**
 * @brief Gets current decimal point character used by the object
 * @param[in] obj   calc structure pointer
 * @return Pointer to the string containing decimal point
 */
const char *calc_get_dec_point_str(calc *obj);

#endif /* __CALC_H__ */
