/*
 * Copyright 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef DISPLAY_H_
#define DISPLAY_H_

#include "utils/types.h"

/**
 * @brief Data structure of display object
 * Used for number I/O and rounding
 */
typedef struct
{
    char str[DISP_STR_SIZE]; /* string buffer of the display */
    int len; /* length of the string in the buffer */
    int dig_count; /* count of decimal digits in the string */
    bool is_neg; /* is number in the string negative */
    bool has_dec_point; /* does number in the string contains decimal point */

    notify_cb change_cb; /* call-back to be invoked on display string change */
    void *change_cb_data; /* data pointer for the call-back */

    char dec_point_char; /* current decimal point character used by the display */

} display;

/**
 * @brief Resets display to "0"
 * @param[in] obj   Display structure pointer
 */
void display_reset(display *obj);

/**
 * @brief Sets new decimal pointer character to use
 * @param[in] obj                Display structure pointer
 * @param[in] new_dec_point_char New character to use as a decimal point
 */
void display_set_dec_point_char(display *obj, char new_dec_point_char);

/**
 * @brief Sets double value number to the display.
 * Number will be rounded according to the DISP_MAX_DIGITS value
 * @param[in] obj   Display structure pointer
 * @param[in] value Double number to set to the display
 * @return Result of the operation. If fail - object's state is not changed
 */
result_type display_set_number(display *obj, double value);

/**
 * @brief Gets number from display as double value
 * @param[in] obj   Display structure pointer
 * @return Double number of the string on display or 0.0 if it is not a number
 */
double display_get_number(display *obj);

/**
 * @brief Processes number key category to enter new characters to display
 * @param[in] obj       Display structure pointer
 * @param[in] key key   Value to process
 * @param[in] reset     True will indicate that display should be reset before key process
 * @return Result of the operation. If fail - object's state is not changed
 */
result_type display_enter_key(display *obj, key_type key, bool reset);

/**
 * @brief Erases last entered character
 * @param[in] obj   Display structure pointer
 * @return Result of the operation. If fail - object's state is not changed
 */
result_type display_erase(display *obj);

/**
 * @brief Negates number on display
 * @param[in] obj   Display structure pointer
 */
void display_negate(display *obj);

/**
 * @brief Sets string message to display.
 * After this function display should be reset to work correctly
 * @param[in] obj       Display structure pointer
 * @param[in] new_str   String to set to the display
 */
void display_set_str(display *obj, const char *new_str);

#endif /* DISPLAY_H_ */
