/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __EDIT_CONTACT_H__
#define __EDIT_CONTACT_H__

#include "common.h"

#include <contacts.h>
#include <time.h>

typedef struct
{
    contacts_record_h contact_record;
    bool is_new_contact;
} sc_ec_contact_edit;


/**
 * @brief Construct new structure for creating new contact
 * @return new sc_ec_contact_edit structure on success, otherwise NULL
 */
sc_ec_contact_edit *sc_ec_create_new_contact();

/**
 * @brief Construct new structure for editing contact
 * @param[in]   contact_id      ID of contact to edit
 * @return new sc_ec_contact_edit structure on success, otherwise NULL
 */
sc_ec_contact_edit *sc_ec_create_edit_contact(int contact_id);

/**
 * @brief Destruct edit contact sturcture
 * param[in]    edit_contact    Structure to destruct
 */
void sc_ec_destruct_edit_contact(sc_ec_contact_edit *edit_contact);

/**
 * @brief Retrieve contact's string value
 * @param[in]   field_type      Type of field: name, number
 * @param[in]   contact_record  Contact's record handle
 * @param[out]  value           String value to return
 * @return true if success, otherwise false
 */
bool sc_ec_get_contact_field_str_val(sc_common_contact_fields field_type, contacts_record_h contact_record, char **value);

/**
 * @brief Set contact's string value
 * @param[in]   field_type      Type of field: name, number
 * @param[in]   contact_record  Contact's record handle
 * @param[in]   value           Passed string value
 * @return true if success, otherwise false
 */
bool sc_ec_set_contact_field_str_val(sc_common_contact_fields field_type, contacts_record_h contact_record, const char *value);

/**
 * @brief Retrieve contact's integer value
 * @param[in]   field_type      Type of field: name, number
 * @param[in]   contact_record  Contact's record handle
 * @param[out]  value           Integer value to return
 * @return true if success, otherwise false
 */
bool sc_ec_get_contact_field_int_val(sc_common_contact_fields field_type, contacts_record_h contact_record, int *value);

/**
 * @brief Set contact's integer value
 * @param[in]   field_type      Type of field: name, number
 * @param[in]   contact_record  Contact's record handle
 * @param[in]   value           Passed integer value
 * @return true if success, otherwise false
 */
bool sc_ec_set_contact_field_int_val(sc_common_contact_fields field_type, contacts_record_h contact_record, int value);

/**
 * @brief Save contact
 * @param[in]   edit_contact    Contact edit structure
 * @return id of created/updated contact on success, otherwise 0
 */
int sc_ec_save_contact(sc_ec_contact_edit *edit_contact);

/**
 * @brief Check whether filled first or last name
 * @param[in]   contact_record      Contact's record handle
 * @return true if exist, otherwise false
 */
bool sc_ec_is_exist_first_or_last_name(contacts_record_h contact_record);

#endif //__EDIT_CONTACT_H__
