/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#include "main-app.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/person-list-view.h"
#include "view/window.h"

#include <app.h>
#include <Elementary.h>

struct _app_data
{
    window_obj  *win;
    Evas_Object *navi;
};

// app event callbacks
static bool _app_create_cb(void *user_data);
static void _app_terminate_cb(void *user_data);

static Evas_Object *_app_navi_add(app_data *app);
static void _app_navi_back_cb(void *data, Evas_Object *obj, void *event_info);

app_data *app_create()
{
    app_data *app = calloc(1, sizeof(app_data));
    if(app)
    {
        if(!sc_db_utils_connect())
        {
            app_destroy(app);
            ERR("Failed to init DB");
            app = NULL;
        }
    }
    return app;
}

void app_destroy(app_data *app)
{
    if(app)
    {
        free(app);
        sc_db_utils_disconnect();
    }
}

int app_run(app_data *app, int argc, char **argv)
{
    RETVM_IF(NULL == app, -1, "app is null");

    ui_app_lifecycle_callback_s cbs =
    {
        .create = _app_create_cb,
        .terminate = _app_terminate_cb,
        NULL
    };
    return ui_app_main(argc, argv, &cbs, app);
}

static Evas_Object *_app_navi_add(app_data *app)
{
    Evas_Object *result = NULL;
    Evas_Object *parent = win_get_host_layout(app->win);
    if(parent)
    {
        result = ui_utils_navi_add(parent, _app_navi_back_cb, app);
        if(result)
        {
            win_set_layout(app->win, result);
        }
    }
    return result;
}

static bool _app_create_cb(void *user_data)
{
    app_data *ad = user_data;
    RETVM_IF(NULL == ad, false, "ad is null");

    ad->win = win_create();
    RETVM_IF(NULL == ad->win, false, "win is null");

    ad->navi = _app_navi_add(ad);
    RETVM_IF(NULL == ad->navi, false, "navi is null");

    sc_plist_view_add(ad->navi);
    return true;
}

static void _app_navi_back_cb(void *data, Evas_Object *obj, void *event_info)
{
    app_data *app = (app_data *)data;

    if(elm_naviframe_top_item_get(obj) == elm_naviframe_bottom_item_get(obj))
    {
        if(app)
        {
            //Collapse application window. To exit use ui_app_exit() instead
            win_lower(app->win);
        }
    }
    else
    {
        elm_naviframe_item_pop(obj);
    }
}

static void _app_terminate_cb(void *user_data)
{
    app_data *app = user_data;
    RETM_IF(NULL == app, "app is null");

    win_destroy(app->win);
    app->win = NULL;
}
