/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/group-list-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-list-utils.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/create-group-view.h"
#include "view/group-details-view.h"
#include "view/view-common.h"

#include <Elementary.h>
#include <contacts.h>

#define ITEM_TEXT_LEN 256

#define GROUP_LIST_VIEW "group_list_view"

static const char *_caption_categories = "Categories";

typedef struct
{
    Evas_Object *navi;
    Elm_Object_Item *navi_item;
    Evas_Object *list;

    contacts_list_h group_list;
} group_list_view;

static void _group_list_view_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info);
static bool _group_list_view_load_groups(group_list_view *view);
static void _group_list_view_fill(group_list_view *view);
static void _group_list_view_sel_cb(void *data, Evas_Object *obj, void *item);
static void _group_list_view_changed_cb(const char *view_uri, void *user_data);

static void _group_list_view_add_create_btn(group_list_view *view);
static void _group_list_view_create_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info);

void sc_group_list_view_add(Evas_Object *navi)
{
    group_list_view *view = calloc(1, sizeof(group_list_view));
    RETM_IF(!view, "calloc() failed");
    view->navi = navi;

    view->list = elm_list_add(view->navi);
    if(!view->list)
    {
        ERR("elm_list_add() failed");
        free(view);
        return;
    }

    evas_object_event_callback_add(view->list, EVAS_CALLBACK_FREE,
            _group_list_view_destroy_cb, view);

    if(!_group_list_view_load_groups(view))
    {
        evas_object_del(view->list);
        return;
    }

    sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_group._uri, _group_list_view_changed_cb, view));
    sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_group_relation._uri, _group_list_view_changed_cb, view));

    _group_list_view_fill(view);

    view->navi_item = elm_naviframe_item_push(view->navi, _caption_categories, NULL, NULL, view->list, NULL);
    evas_object_data_set(view->navi, NAVI_ITEM_CATEGORIES_LIST, view->navi_item);

    _group_list_view_add_create_btn(view);
}

static void _group_list_view_destroy_cb(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
    RETM_IF(!data, "data is NULL");
    group_list_view *view = data;
    contacts_db_remove_changed_cb(_contacts_group._uri, _group_list_view_changed_cb, data);
    contacts_db_remove_changed_cb(_contacts_group_relation._uri, _group_list_view_changed_cb, view);
    evas_object_data_del(view->navi, NAVI_ITEM_CATEGORIES_LIST);
    contacts_list_destroy(view->group_list, true);
    free(view);
}

static bool _group_list_view_load_groups(group_list_view *view)
{
    RETVM_IF(!view, false, "view is NULL");
    bool success = sc_db_utils_is_success(contacts_db_get_all_records(_contacts_group._uri,
            0, 0, &view->group_list));
    return success;
}

static void _group_list_view_fill(group_list_view *view)
{
    RETM_IF(!view, "view is NULL");

    contacts_record_h group;
    CONTACTS_LIST_FOREACH(view->group_list, group)
    {
        char *group_name = NULL;
        int group_id = 0;
        int count = 0;

        bool success = sc_db_utils_is_success(
                contacts_record_get_str_p(group, _contacts_group.name, &group_name))
            && sc_db_utils_is_success(
                contacts_record_get_int(group, _contacts_group.id, &group_id));
        if(!success)
        {
            continue;
        }

        contacts_list_h members = sc_list_utils_get_records_list_int_filter(_contacts_group_relation._uri,
                _contacts_group_relation.group_id, group_id);
        if(members)
        {
            contacts_list_get_count(members, &count);
            contacts_list_destroy(members, true);
        }
        else
        {
            ERR("sc_list_get_records_list_int_filter() failed");
        }

        char buffer[ITEM_TEXT_LEN] = { '\0' };
        snprintf(buffer, sizeof(buffer), "%s (%d)", group_name, count);
        evas_object_data_set(view->list, GROUP_LIST_VIEW, view);
        elm_list_item_append(view->list, buffer, NULL, NULL, _group_list_view_sel_cb, (void *)group_id);
    }
    elm_list_go(view->list);
}

static void _group_list_view_sel_cb(void *data, Evas_Object *obj, void *item)
{
    RETM_IF(!data, "data is NULL");
    RETM_IF(!obj, "obj is NULL");
    elm_list_item_selected_set(item, EINA_FALSE);

    group_list_view *group_data = evas_object_data_get(obj, GROUP_LIST_VIEW);
    sc_group_details_view_add(group_data->navi, (int) data);
}

static void _group_list_view_changed_cb(const char *view_uri, void *user_data)
{
    RETM_IF(!user_data, "user_data is NULL");
    group_list_view *view = user_data;

    contacts_list_destroy(view->group_list, true);
    _group_list_view_load_groups(view);
    elm_list_clear(view->list);
    _group_list_view_fill(view);
}

static void _group_list_view_add_create_btn(group_list_view *view)
{
    RETM_IF(!view, "data is NULL");
    RETM_IF(!view->navi, "parent is NULL");
    RETM_IF(!view->navi_item, "navi item is NULL");

    Evas_Object *toolbar = ui_utils_toolbar_add(view->list);
    if(toolbar)
    {
        elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
        elm_toolbar_item_append(toolbar, NULL, _caption_create, _group_list_view_create_btn_clicked_cb, view);
        elm_object_item_part_content_set(view->navi_item, PART_TOOLBAR, toolbar);
    }
}

static void _group_list_view_create_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(!data, "data is NULL");
    group_list_view *view = data;
    sc_gcreate_view_add(view->navi);
}
