/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "view/person-list-view.h"

#include "model/contacts-utils/contacts-db-utils.h"
#include "model/contacts-utils/contacts-list-utils.h"
#include "utils/logger.h"
#include "utils/ui-utils.h"
#include "view/contact-details-view.h"
#include "view/edit-contact-view.h"
#include "view/favourites-view.h"
#include "view/group-list-view.h"
#include "view/view-common.h"

#include <contacts.h>
#include <efl_extension.h>
#include <Elementary.h>


#define DATA_LIST                   "data_list"

static const char *_caption_contacts            = "Contacts";
static const char *_caption_categories          = "Categories";
static const char *_caption_favourites          = "Favourites";
static const char *_three_dots                  = "...";

static const int _reduced_name_length = 20;
static const int _name_max_length = 17;


typedef struct
{
    Evas_Object *navi;
    Elm_Object_Item *navi_item;
    Evas_Object *list;
    Evas_Object *menu;
} list_view_data;

static void _plist_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info);
static void _plist_view_fill(list_view_data *data);
static void _plist_view_sel_cb(void *data, Evas_Object *obj, void *event_info);

static void _plist_view_update_list_cb(const char *view_uri, void *user_data);

static Evas_Object *_plist_view_menu_create(list_view_data *data);
static void _plist_view_del_object_cb(void *data, Evas_Object *obj, void *event_info);
static void _plist_view_menu_cb(void *data, Evas_Object *obj, void *event_info);

static void _plist_view_categories_cb(void *data, Evas_Object *obj, void *event_info);
static void _plist_view_favourites_cb(void *data, Evas_Object *obj, void *event_info);

static void _plist_view_add_create_btn(list_view_data *data);
static void _plist_view_create_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info);


void sc_plist_view_add(Evas_Object *parent)
{
    RETM_IF(NULL == parent, "parent is null");

    list_view_data *data = calloc(1, sizeof(list_view_data));
    RETM_IF(!data, "calloc() failed");
    data->navi = parent;

    data->list = elm_list_add(data->navi);
    if(!data->list)
    {
        ERR("elm_list_add() failed");
        free(data);
        return;
    }

    _plist_view_fill(data);

    evas_object_event_callback_add(data->list, EVAS_CALLBACK_FREE, _plist_view_destroy, data);
    sc_db_utils_is_success(contacts_db_add_changed_cb(_contacts_person._uri, _plist_view_update_list_cb, data));

    data->navi_item = elm_naviframe_item_push(data->navi, _caption_contacts, NULL, NULL, data->list, NULL);
    evas_object_data_set(data->navi, NAVI_ITEM_PERSON_LIST, data->navi_item);

    Evas_Object *more_btn = elm_button_add(data->navi);
    evas_object_smart_callback_add(more_btn, EVENT_CLICKED, _plist_view_menu_cb, data);
    elm_object_item_part_content_set(data->navi_item, PART_TOOLBAR_MORE_BUTTON, more_btn);

    _plist_view_add_create_btn(data);
}

static void _plist_view_destroy(void *data, Evas *e, Evas_Object *obj, void *event_info)
{
    RETM_IF(NULL == data, "data is null");
    contacts_db_remove_changed_cb(_contacts_person._uri, _plist_view_update_list_cb, data);
    free(data);
}

static Evas_Object *_plist_view_menu_create(list_view_data *data)
{
    RETVM_IF(NULL == data, NULL, "data is null");
    Evas_Object *menu = elm_ctxpopup_add(data->navi);
    eext_object_event_callback_add(menu, EEXT_CALLBACK_BACK, _plist_view_del_object_cb, data);
    eext_object_event_callback_add(menu, EEXT_CALLBACK_MORE, _plist_view_del_object_cb, data);
    elm_ctxpopup_item_append(menu, _caption_categories, NULL, _plist_view_categories_cb, data);
    elm_ctxpopup_item_append(menu, _caption_favourites, NULL, _plist_view_favourites_cb, data);

    int y = 0;
    int height = 0;
    evas_object_geometry_get(data->list, NULL, &y, NULL, &height);
    evas_object_move(menu, 0, y + height);
    return menu;
}

static void _plist_view_del_object_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(!data, "data is null");
    list_view_data *plist_data = data;
    evas_object_del(plist_data->menu);
    plist_data->menu = NULL;
}

static void _plist_view_fill(list_view_data *data)
{
    RETM_IF(NULL == data, "data is null");
    contacts_record_h record = NULL;

    contacts_list_h person_list = NULL;
    if(sc_db_utils_is_success(contacts_db_get_all_records(_contacts_person._uri, 0, 0, &person_list)))
    {
        CONTACTS_LIST_FOREACH(person_list, record)
        {
            char *name = NULL;
            int id = 0;
            if( sc_db_utils_is_success(contacts_record_get_str_p(record, _contacts_person.display_name, &name))
             && sc_db_utils_is_success(contacts_record_get_int(record, _contacts_person.id, &id)) )
            {
                if (name && (strlen(name) >= _reduced_name_length))
                {
                    char *reduced_name = calloc(_reduced_name_length + 1, sizeof(char));
                    strncat(reduced_name, name, _name_max_length);
                    strcat(reduced_name, _three_dots);
                    char *temp_str = elm_entry_utf8_to_markup(reduced_name);
                    elm_list_item_append(data->list, temp_str, NULL, NULL, _plist_view_sel_cb, (void *)id);
                    free(temp_str);
                    free(reduced_name);
                }
                else
                {
                    char *temp_str = elm_entry_utf8_to_markup(name);
                    elm_list_item_append(data->list, temp_str, NULL, NULL, _plist_view_sel_cb, (void *)id);
                    free(temp_str);
                }
                evas_object_data_set(data->list, DATA_LIST, data);
            }
        }
        contacts_list_destroy(person_list, true);
    }
    elm_list_go(data->list);
}

static void _plist_view_sel_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(NULL == data, "data is null");
    RETM_IF(NULL == obj, "obj is null");
    RETM_IF(NULL == event_info, "event_info is null");
    list_view_data *list_data = evas_object_data_get(obj, DATA_LIST);
    RETM_IF(NULL == list_data, "list data is null");

    elm_list_item_selected_set(event_info, EINA_FALSE);

    sc_cdetails_view_add(list_data->navi, (int)data);
}

static void _plist_view_update_list_cb(const char *view_uri, void *user_data)
{
    RETM_IF(NULL == user_data, "user_data is null");
    list_view_data *data = user_data;
    elm_list_clear(data->list);
    _plist_view_fill(data);
}

static void _plist_view_menu_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(NULL == data, "data is null");
    list_view_data *list_data = data;

    if(evas_object_visible_get(list_data->menu))
    {
        elm_ctxpopup_dismiss(list_data->menu);
    }
    else
    {
        if(!list_data->menu)
        {
            list_data->menu = _plist_view_menu_create(list_data);
            RETM_IF(!list_data->menu, "_plist_view_menu_create() failed");
        }

        evas_object_show(list_data->menu);
    }
}

static void _plist_view_categories_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(NULL == data, "data is null");
    RETM_IF(NULL == obj, "obj is null");
    list_view_data *list_data = data;
    evas_object_del(obj);
    list_data->menu = NULL;
    sc_group_list_view_add(list_data->navi);
}

static void _plist_view_favourites_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(NULL == data, "data is null");
    RETM_IF(NULL == obj, "obj is null");
    list_view_data *list_data = data;
    evas_object_del(obj);
    list_data->menu = NULL;
    sc_favourites_view_add(list_data->navi);
}

static void _plist_view_add_create_btn(list_view_data *data)
{
    RETM_IF(NULL == data, "data is null");
    RETM_IF(NULL == data->navi, "navi is null");
    RETM_IF(NULL == data->navi_item, "navi item is null");

    Evas_Object *toolbar = ui_utils_toolbar_add(data->navi);
    elm_toolbar_select_mode_set(toolbar, ELM_OBJECT_SELECT_MODE_NONE);
    elm_toolbar_item_append(toolbar, NULL, _caption_create, _plist_view_create_btn_clicked_cb, data);
    elm_object_item_part_content_set(data->navi_item, PART_TOOLBAR, toolbar);
}

static void _plist_view_create_btn_clicked_cb(void *data, Evas_Object *obj, void *event_info)
{
    RETM_IF(NULL == data, "data is null");
    list_view_data *lv_data = data;
    sc_ec_view_add(lv_data->navi, 0);
}
