/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "ecore_timer"
#include "main.h"

#define TIMEOUT_1 1.0 // interval for timer1
#define TIMEOUT_2 3.0 // timer2 - delay timer1
#define TIMEOUT_3 8.2 // timer3 - pause timer1
#define TIMEOUT_4 11.0 // timer4 - resume timer1
#define TIMEOUT_5 14.0 // timer5 - change interval of timer1
#define TIMEOUT_6 18.0 // top timer1 and start timer7 and timer8 with changed precision
#define TIMEOUT_7 1.1 // interval for timer7
#define TIMEOUT_8 1.2 // interval for timer8
#define DELAY_1   3.0 // delay time for timer1 - used by timer2
#define INTERVAL1 2.0 // new interval for timer1 - used by timer5

struct context
{
	Ecore_Timer *timer1;
	Ecore_Timer *timer2;
	Ecore_Timer *timer3;
	Ecore_Timer *timer4;
	Ecore_Timer *timer5;
	Ecore_Timer *timer6;
	Ecore_Timer *timer7;
	Ecore_Timer *timer8;
};

static double _initial_time = 0;
static struct context ctxt = {0};

static double
_get_current_time(void)
{
	return ecore_time_get() - _initial_time;
}

static Eina_Bool
_timer1_cb(void *data EINA_UNUSED)
{
	dlog_print(DLOG_INFO, LOG_TAG, "Timer1 expired after %0.3f seconds.", _get_current_time());
	return ECORE_CALLBACK_RENEW;
}

static Eina_Bool
_timer2_cb(void *data EINA_UNUSED)
{
	struct context *ctxt = data;
	dlog_print(DLOG_INFO, LOG_TAG, "Timer2 expired after %0.3f seconds.", _get_current_time());
	dlog_print(DLOG_INFO, LOG_TAG, "Adding delay of %0.3f seconds to timer1.", DELAY_1);

	ecore_timer_delay(ctxt->timer1, DELAY_1);

	ctxt->timer2 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
_timer3_cb(void *data)
{
	struct context *ctxt = data;
	dlog_print(DLOG_INFO, LOG_TAG, "Timer3 expired after %0.3f seconds.", _get_current_time());
	dlog_print(DLOG_INFO, LOG_TAG, "Freezing timer1.");

	ecore_timer_freeze(ctxt->timer1);

	ctxt->timer3 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
_timer4_cb(void *data)
{
	struct context *ctxt = data;
	dlog_print(DLOG_INFO, LOG_TAG, "Timer4 expired after %0.3f seconds.", _get_current_time());
	dlog_print(DLOG_INFO, LOG_TAG, "Resuming timer1, which has %0.3f seconds left to expire.", ecore_timer_pending_get(ctxt->timer1));

	ecore_timer_thaw(ctxt->timer1);

	ctxt->timer4 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
_timer5_cb(void *data)
{
	struct context *ctxt = data;
	double interval = ecore_timer_interval_get(ctxt->timer1);

	dlog_print(DLOG_INFO, LOG_TAG, "Timer5 expired after %0.3f seconds.", _get_current_time());
	dlog_print(DLOG_INFO, LOG_TAG, "Changing interval of timer1 from %0.3f to %0.3f seconds.", interval, INTERVAL1);

	ecore_timer_interval_set(ctxt->timer1, INTERVAL1);

	ctxt->timer5 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
_timer7_cb(void *data)
{
	struct context *ctxt = data;
	dlog_print(DLOG_INFO, LOG_TAG, "Timer7 expired after %0.3f seconds.", _get_current_time());

	ctxt->timer7 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
_timer8_cb(void *data)
{
	struct context *ctxt = data;
	dlog_print(DLOG_INFO, LOG_TAG, "Timer8 expired after %0.3f seconds.", _get_current_time());

	ctxt->timer8 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
_timer6_cb(void *data)
{
	struct context *ctxt = data;
	dlog_print(DLOG_INFO, LOG_TAG, "Timer6 expired after %0.3f seconds.", _get_current_time());

	dlog_print(DLOG_INFO, LOG_TAG, "Stopping timer1.");

	ecore_timer_del(ctxt->timer1);
	ctxt->timer1 = NULL;

	dlog_print(DLOG_INFO, LOG_TAG, "Starting timer7 (%0.3fs) and timer8 (%0.3fs).", TIMEOUT_7, TIMEOUT_8);

	ctxt->timer7 = ecore_timer_add(TIMEOUT_7, _timer7_cb, ctxt);
	ctxt->timer8 = ecore_timer_add(TIMEOUT_8, _timer8_cb, ctxt);

	ecore_timer_precision_set(0.2);

	ctxt->timer6 = NULL;
	return ECORE_CALLBACK_CANCEL;
}

static Eina_Bool
naviframe_pop_cb(void *data, Elm_Object_Item *it)
{
	ecore_timer_del(ctxt.timer1);
	ecore_timer_del(ctxt.timer2);
	ecore_timer_del(ctxt.timer3);
	ecore_timer_del(ctxt.timer4);
	ecore_timer_del(ctxt.timer5);
	ecore_timer_del(ctxt.timer6);
	ecore_timer_del(ctxt.timer7);
	ecore_timer_del(ctxt.timer8);

	return EINA_TRUE;
}

void
ecore_timer_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	Elm_Object_Item *nf_it;

	/* Base Layout */
	ad->layout = elm_layout_add(ad->win);
	elm_layout_theme_set(ad->layout, "layout", "application", "default");
	evas_object_size_hint_weight_set(ad->layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(ad->layout);

	/* Label*/
	Evas_Object *label = elm_label_add(ad->layout);
	elm_object_text_set(label, "Ecore Timer");
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(ad->layout, "elm.swallow.content", label);
	evas_object_show(label);

	/* Timer */
	_initial_time = ecore_time_get();
	ctxt.timer1 = ecore_timer_add(TIMEOUT_1, _timer1_cb, &ctxt);
	ctxt.timer2 = ecore_timer_add(TIMEOUT_2, _timer2_cb, &ctxt);
	ctxt.timer3 = ecore_timer_add(TIMEOUT_3, _timer3_cb, &ctxt);
	ctxt.timer4 = ecore_timer_add(TIMEOUT_4, _timer4_cb, &ctxt);
	ctxt.timer5 = ecore_timer_add(TIMEOUT_5, _timer5_cb, &ctxt);
	ctxt.timer6 = ecore_timer_add(TIMEOUT_6, _timer6_cb, &ctxt);

	dlog_print(DLOG_INFO, LOG_TAG, "start the main loop.");

	/* Push base layout into naviframe */
	nf_it = elm_naviframe_item_push(ad->nf, NULL, NULL, NULL, ad->layout, "empty");
	elm_naviframe_item_pop_cb_set(nf_it, naviframe_pop_cb, ad);
}
