/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "eina_stringshare"
#include "main.h"

static Eina_Bool
naviframe_pop_cb(void *data, Elm_Object_Item *it)
{
	return EINA_TRUE;
}

void
eina_stringshare_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;

	Elm_Object_Item *nf_it;

	/* Base Layout */
	ad->layout = elm_layout_add(ad->win);
	elm_layout_theme_set(ad->layout, "layout", "application", "default");
	evas_object_size_hint_weight_set(ad->layout, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_show(ad->layout);

	/* Label*/
	Evas_Object *label = elm_label_add(ad->layout);
	elm_object_text_set(label, "Eina Stringshare");
	evas_object_size_hint_weight_set(label, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_object_part_content_set(ad->layout, "elm.swallow.content", label);
	evas_object_show(label);

	/*
	 * StringShare sample
	 */
	const char *str1, *str2;

	// We start the substantive part of the example by showing how to make a part of a string shared and how to get the length of a shared string:

	const char *prologe1 = "The Cylons were created by man. They rebelled. They evolved.";
	str1 = eina_stringshare_add_length(prologe1, 31);
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s", str1);
	int length = eina_stringshare_strlen(str1);
	dlog_print(DLOG_DEBUG, LOG_TAG, "length : %d", length);

	// As we add shared strings we also need to delete them when done using them:

	eina_stringshare_del(str1);
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s", str1);

	// There are many ways of creating shared strings including an equivalent to sprintf:

	const char *prologe2 = "%d Cylon models. %d are known. %d live in secret. %d will be revealed.";
	str1 = eina_stringshare_printf(prologe2, 12, 7, 4, 1);
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s", str1);
	eina_stringshare_del(str1);

	// An equivalent to snprintf:

	const char *prologe3 = "There are many opies. And they have a plan";
	str1 = eina_stringshare_nprintf(45, "%s", prologe3);
	length = eina_stringshare_strlen(str1);

	// But the simplest way of creating a shared string is through eina_stringshare_add();

	dlog_print(DLOG_DEBUG, LOG_TAG, "str1 : %s", str1);

	// Sometimes you already have a pointer to a shared string and want to use it, so to make sure the provider of the pointer won't free it while you're using it you can increase the shared string's ref count:

	str2 = eina_stringshare_add(prologe3);
	length = eina_stringshare_strlen(str1);

	dlog_print(DLOG_DEBUG, LOG_TAG, "str2 : %s", str2);

	// Whenever you have a pointer to a shared string and would like to change it's value you should use eina_stringshare_replace():

	eina_stringshare_ref(str2);
	eina_stringshare_del(str2);
	dlog_print(DLOG_DEBUG, LOG_TAG, "%s", str2);

	/* Push base layout into naviframe */
	nf_it = elm_naviframe_item_push(ad->nf, NULL, NULL, NULL, ad->layout, "empty");
	elm_naviframe_item_pop_cb_set(nf_it, naviframe_pop_cb, ad);
}
