/*
 * Copyright (c) 2014 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "$(appName).h"

typedef struct appdata{
	Evas_Object* win;
	Evas_Object* conform;
	Evas_Object* naviframe;
	unsigned int count;
} appdata_s;

static void
win_delete_request_cb(void *data , Evas_Object *obj , void *event_info)
{
	ui_app_exit();
}

static void
nf_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	appdata_s *ad = data;
	/* Let window go to hide state. */
	elm_win_lower(ad->win);
}

static void
entry_language_changed_cb(void *data, Evas_Object *obj, void *event_info)
{
	/* Get the language changed callback call by widget.
	   If you need to apply some markups to text,
	   then use the i18n_get_text() in the language,changed smart callback. */
	appdata_s *ad = data;
	char buf[128];
	snprintf(buf, sizeof(buf), "%s: %d", i18n_get_text("IDS_CURRENT"), ad->count);
	elm_entry_entry_set(obj, buf);
}

static char *
gl_text_get_cb(void *data, Evas_Object *obj, const char *part)
{
	/* In case of genlist, the text will be updated in realtime when language is changed.
	   No need to use any Elementary translatable_text_set() APIs. */
	return strdup(i18n_get_text("IDS_SELECT_ITEM"));
}

static void
create_view(appdata_s *ad)
{
	Evas_Object *box, *btn, *entry, *genlist;
	Elm_Object_Item *nf_it;
	Elm_Genlist_Item_Class *itc;
	char buf[128];
	int i;

	/* Box */
	box = elm_box_add(ad->naviframe);

	/* Entry */
	entry = elm_entry_add(box);
	evas_object_smart_callback_add(entry, "language,changed", entry_language_changed_cb, ad);
	evas_object_size_hint_align_set(entry, EVAS_HINT_FILL, 0);
	/* Basically, you can use i18n_get_text() to get the current language text. */
	snprintf(buf, sizeof(buf), "%s: %d", i18n_get_text("IDS_CURRENT"), ad->count);
	elm_entry_entry_set(entry, buf);
	elm_entry_editable_set(entry, EINA_FALSE);
	evas_object_show(entry);
	elm_box_pack_end(box, entry);

	/* Genlist */
	genlist = elm_genlist_add(box);

	itc = elm_genlist_item_class_new();
	itc->item_style = "default";
	itc->func.text_get = gl_text_get_cb;

	for (i = 0; i < 100; i++) {
		elm_genlist_item_append(genlist, itc, NULL, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	}

	elm_genlist_item_class_free(itc);

	evas_object_size_hint_weight_set(genlist, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(genlist, EVAS_HINT_FILL, EVAS_HINT_FILL);
	evas_object_show(genlist);
	elm_box_pack_end(box, genlist);

	/* Button */
	btn = elm_button_add(box);
	/* Basically, you can use elm_object_translatable_part_text_set() API
	   to support dynamic language change. */
	elm_object_translatable_text_set(btn, "IDS_DONE");
	evas_object_size_hint_align_set(btn, EVAS_HINT_FILL, 1);
	evas_object_show(btn);
	elm_box_pack_end(box, btn);

	/* Push a view to naviframe */
	nf_it = elm_naviframe_item_push(ad->naviframe, "IDS_LANGUAGE", NULL, NULL, box, NULL);
	/* Basically, you can use
		elm_object_item_part_text_translatable_set() or
		elm_object_item_translatable_part_text_set() API to support
		dynamic language change.*/
	elm_object_item_part_text_translatable_set(nf_it, NULL, EINA_TRUE);
}

static void
create_base_gui(appdata_s *ad)
{
	/* Window */
	ad->win = elm_win_util_standard_add(PACKAGE, PACKAGE);
	elm_win_conformant_set(ad->win, EINA_TRUE);
	elm_win_autodel_set(ad->win, EINA_TRUE);

	if (elm_win_wm_rotation_supported_get(ad->win)) {
		int rots[4] = { 0, 90, 180, 270 };
		elm_win_wm_rotation_available_rotations_set(ad->win, (const int *)(&rots), 4);
	}

	evas_object_smart_callback_add(ad->win, "delete,request", win_delete_request_cb, NULL);

	/* Conformant */
	ad->conform = elm_conformant_add(ad->win);
	evas_object_size_hint_weight_set(ad->conform, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	elm_win_resize_object_add(ad->win, ad->conform);
	evas_object_show(ad->conform);

	/* Naviframe */
	ad->naviframe = elm_naviframe_add(ad->conform);
	eext_object_event_callback_add(ad->naviframe, EEXT_CALLBACK_BACK, nf_back_cb, ad);
	elm_object_content_set(ad->conform, ad->naviframe);

	/* Show window after base gui is set up */
	evas_object_show(ad->win);
}

static Eina_Bool
timer_cb(void *data)
{
	appdata_s *ad = data;
	ad->count++;
	return ECORE_CALLBACK_RENEW;
}

static bool
app_create(void *data)
{
	/* Hook to take necessary actions before main event loop starts
	   Initialize UI resources and application's data
	   If this function returns true, the main loop of application starts
	   If this function returns false, the application is terminated */
	appdata_s *ad = data;

	create_base_gui(ad);
	create_view(ad);

	ecore_timer_add(1, timer_cb, ad);

	return true;
}

static void
app_control(app_control_h app_control, void *data)
{
	/* Handle the launch request. */
}

static void
app_pause(void *data)
{
	/* Take necessary actions when application becomes invisible. */
}

static void
app_resume(void *data)
{
	/* Take necessary actions when application becomes visible. */
}

static void
app_terminate(void *data)
{
	/* Release all resources. */
}

static void
ui_app_lang_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LANGUAGE_CHANGED*/
	char *locale = NULL;
	system_settings_get_value_string(SYSTEM_SETTINGS_KEY_LOCALE_LANGUAGE, &locale);
	elm_language_set(locale);
	free(locale);
	return;
}

static void
ui_app_orient_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_DEVICE_ORIENTATION_CHANGED*/
	return;
}

static void
ui_app_region_changed(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_REGION_FORMAT_CHANGED*/
}

static void
ui_app_low_battery(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_BATTERY*/
}

static void
ui_app_low_memory(app_event_info_h event_info, void *user_data)
{
	/*APP_EVENT_LOW_MEMORY*/
}

int
main(int argc, char *argv[])
{
	appdata_s ad= {0,};
	int ret = 0;

	ui_app_lifecycle_callback_s event_callback = {0,};
	app_event_handler_h handlers[5] = {NULL, };

	event_callback.create = app_create;
	event_callback.terminate = app_terminate;
	event_callback.pause = app_pause;
	event_callback.resume = app_resume;
	event_callback.app_control = app_control;

	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_BATTERY], APP_EVENT_LOW_BATTERY, ui_app_low_battery, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LOW_MEMORY], APP_EVENT_LOW_MEMORY, ui_app_low_memory, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_DEVICE_ORIENTATION_CHANGED], APP_EVENT_DEVICE_ORIENTATION_CHANGED, ui_app_orient_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_LANGUAGE_CHANGED], APP_EVENT_LANGUAGE_CHANGED, ui_app_lang_changed, &ad);
	ui_app_add_event_handler(&handlers[APP_EVENT_REGION_FORMAT_CHANGED], APP_EVENT_REGION_FORMAT_CHANGED, ui_app_region_changed, &ad);
	ui_app_remove_event_handler(handlers[APP_EVENT_LOW_MEMORY]);

	ret = ui_app_main(argc, argv, &event_callback, &ad);
	if (ret != APP_ERROR_NONE) {
		dlog_print(DLOG_ERROR, LOG_TAG, "ui_app_main() is failed. err = %d", ret);
	}

	return ret;
}
